<?php

/**
 * Class StaticFormContentBlock
 *
 * @method HasManyList|StaticFormField[] FormFields()
 * @property string $SubmitButtonText
 */
class StaticFormContentBlock extends ContentBlock
{

	protected static $singular_name = 'Static Form Block';
	protected static $plural_name = 'Sign Form Blocks';

	protected static $db = [
		'SubmitButtonText'    => 'Varchar',
		'LabelDisplayStyle'   => 'Varchar',
		'ContentBeforeForm'   => 'HTMLText',
		'ContentAfterForm'    => 'HTMLText',
		'ContentAfterSuccess' => 'HTMLText',
		'EmailAction'         => 'Varchar',
		'CustomEmailAddress'  => 'Varchar',
	];

	protected static $defaults = [
		'EmailAction'         => 'DefaultEmail',
		'SubmitButtonText'    => 'Submit',
		'LabelDisplayStyle'   => 'Label',
		'ContentAfterSuccess' => '<p>Thank you for your submission</p>'
	];

	protected static $has_one = [
		'BackgroundImage' => 'Image'
	];

	protected static $has_many = [
		'FormFields' => 'StaticFormField'
	];

	public function getCMSFields()
	{

		$this->beforeUpdateCMSFields(function (FieldList $fields) {

			if ($this->isInDB()) {
				$fields->removeByName('FormFields');

				$fields->addFieldToTab('Root.Form', DropdownField::create('LabelDisplayStyle', 'Labels', ['Label' => 'Visible', 'Placeholder' => 'Hidden']));
				$fields->addFieldToTab('Root.Form', TextField::create('SubmitButtonText', 'Submit Text'));


				$fields->addFieldToTab('Root.Form', DropdownField::create('EmailAction', 'Email submissions?', [
					'DefaultEmail' => 'To the default email associated with my account',
					'CustomEmail'  => 'To a different email',
					'None'         => 'Do not email submissions',
				]));

				$customEmailAddressField = EmailField::create('CustomEmailAddress', 'Email Address');
				$customEmailAddressField->hideUnless('EmailAction')->isEqualTo("CustomEmail")->end();
				$fields->addFieldToTab('Root.Form', $customEmailAddressField);

				/** @var GridFieldConfig_RecordEditor $config */
				$config = GridFieldConfig_RecordEditor::create();
				$config->removeComponentsByType('GridFieldAddNewButton');
				$config->addComponent(new GridFieldAddNewMultiClass('buttons-before-left'));
				$config->addComponent(new GridFieldOrderableRows('SortOrder'));
				$fields->addFieldToTab('Root.Form', GridField::create('FormFields', 'Form Fields', $this->FormFields(), $config));
			} else {
				$fields->addFieldToTab('Root.Main', LiteralField::create(
					'RequiresSaving',
					'<p class="message warning">You may add fields to the form after saving for the first time.</p>')
				);
			}

			$showBackground = $this->owner->config()->get('show_background');
			$fields->removeByName('BackgroundImage');
			if ($showBackground) {
				$fields->addFieldToTab('Root.Main', UploadField::create('BackgroundImage', 'Background'), 'ContentAfterSuccess');
			}

			$showFormBeforeContent = $this->owner->config()->get('show_form_before_content');
			$fields->removeByName('ContentBeforeForm');
			if ($showFormBeforeContent) {
				$fields->insertBefore('ContentAfterSuccess', HtmlEditorField::create('ContentBeforeForm', 'Content Before Form')->setRows(8));
			}

			$showFormAfterContent = $this->owner->config()->get('show_form_after_content');
			$fields->removeByName('ContentAfterForm');
			if ($showFormAfterContent) {
				$fields->insertBefore('ContentAfterSuccess', HtmlEditorField::create('ContentAfterForm', 'Content After Form')->setRows(8));
			}

			$fields->addFieldsToTab('Root.Main', [
				HtmlEditorField::create('ContentAfterSuccess', 'Content After Successful Submission')->setRows(8),
			]);


		});

		$fields = parent::getCMSFields();

		return $fields;
	}

	public function Form()
	{
		$fields = FieldList::create([
			HiddenField::create('FormClassName', '', $this->ClassName),
			HiddenField::create('FormID', '', $this->ID),
		]);
		$actions = FieldList::create();
		$validator = new RequiredFields();
		foreach ($this->FormFields() as $field) {
			$formField = $field->getFormField();
			if ($formField) {
				$fields->push($formField);
				if ($field->Required) {
					$validator->addRequiredField($formField->getName());
				}
			}
		}

		$actions->push(FormAction::create('Submit', $this->SubmitButtonText));

		$form = new Form(Controller::curr(), 'StaticForm', $fields, $actions, $validator);
		$form->setAttribute('data-hook', 'static-form');
		$submissionRoot = array_flip(Config::inst()->get('Director', 'rules'))['StaticFormController'];
		$form->setFormAction("/{$submissionRoot}/save");

		foreach ($this->FormFields() as $field) {
			if($field->hasMethod('withForm')) {
				$field->withForm($form);
			}
		}

		$this->extend('updateForm', $form);
		return $form;
	}

	public function canAddBlock($owner)
	{
		/** @var DataList $otherFormBlocks */
		$otherFormBlocks = $owner->ContentBlocks()->Filter('ClassName', self::class);
		return !$otherFormBlocks->exists();
	}

}