<?php

class EventsPage extends Page
{

	protected static $singular_name = 'Events Listing Page';
	protected static $plural_name = 'Events Listing Pages';

	protected static $allowed_children = ['EventPage'];

	private static $has_many = [
		'EventCategories' => 'EventCategory.EventsPage',
		'EventLocations'  => 'EventLocation.EventsPage'
	];

	private static $db = [
		"EventsPerPage" => "Int",
	];

	private static $extensions = array(
		"EventsPageFilter",
	);

	public function getCMSFields()
	{

		$this->beforeUpdateCMSFields(function (FieldList $fields) {
			$events = $this->getEvents('past');
			$excluded = $this->getExcludedSiteTreeClassNames();
			if (!empty($excluded)) {
				$events = $events->filter("ClassName", $excluded);
				/** @var GridFieldConfig_Event $config */
				$config = GridFieldConfig_Event::create();
				$config->removeComponentsByType('GridFieldEventAddNewButton');
				$gridField = new GridField("PastEvents", "Past Events", $events, $config);
				$fields->addFieldToTab("Root.PastEvents", $gridField);
			}

			$events = $this->getEvents('upcoming');
			$excluded = $this->getExcludedSiteTreeClassNames();
			if (!empty($excluded)) {
				$events = $events->filter("ClassName", $excluded);
				$config = GridFieldConfig_Event::create();
				$config->getComponentByType('GridFieldEventAddNewButton')->setButtonName('Add New Event');
				$gridField = new GridField("UpcomingEvents", "Upcoming Events", $events, $config);
				$fields->addFieldToTab("Root.UpcomingEvents", $gridField);
			}

			$EventLocationsConfig = new GridFieldConfig_RecordEditor();
			$EventLocationsConfig->removeComponentsByType("GridFieldAddNewButton");
			$EventLocationsConfig->addComponent(new GridFieldAddByDBField("buttons-before-left"));
			$EventLocations = new GridField("EventLocationsGridField", "Locations", $this->EventLocations(), $EventLocationsConfig);
			$fields->addFieldToTab('Root.Locations', $EventLocations);

			$EventCategoriesConfig = new GridFieldConfig_RelationEditor();
			$EventCategoriesConfig->removeComponentsByType("GridFieldAddNewButton");
			$EventCategoriesConfig->addComponent(new GridFieldAddByDBField("buttons-before-left"));
			$EventCategories = new GridField("EventCategoriesGridField", "Categories", $this->EventCategories(), $EventCategoriesConfig);
			$fields->addFieldToTab('Root.Categories', $EventCategories);
		});

		$fields = parent::getCMSFields();

		$fields->removeByName('ContentBlocks');

		return $fields;
	}

	public function getPastEventsLink()
	{
		return $this->Link('past');
	}

	public function getCurrentEventsLink()
	{
		return $this->Link();
	}

	public function getSettingsFields()
	{
		$fields = parent::getSettingsFields();
		$fields->addFieldToTab("Root.Settings", NumericField::create("EventsPerPage", "Events Per Page"));

		return $fields;
	}

	public function getExcludedSiteTreeClassNames()
	{
		$classes = array();
		$tmpClasses = $this->allowedChildren();
		foreach ($tmpClasses as $class) {
			if (!Config::inst()->get($class, "show_in_sitetree")) {
				$classes[$class] = $class;
			}
		}

		return $classes;
	}

	public function getEvents($type = 'all')
	{
		$events = EventPage::get()->filter("ParentID", $this->ID);

		switch (strtolower($type)) {
			case 'upcoming':
			case 'future':
			case 'current':
				$events = $events->filter(['EventEndDate:GreaterThanOrEqual' => SS_Datetime::now()->Date()])->sort('EventStartDate', 'ASC');
				break;
			case 'past':
				$events = $events->filter(['EventEndDate:LessThan' => SS_Datetime::now()->Date()])->sort('EventStartDate', 'DESC');
				break;
		}

		//Allow decorators to manipulate list
		$this->extend('updateGetEvents', $events);

		return $events;
	}
}

class EventsPage_Controller extends Page_Controller
{

	private static $extensions = array(
		"EventSearchExtension"
	);

	private static $allowed_actions = array(
		'past',
		'location',
		'category',
		'rss',
		'EventSearchForm',
		'doSearch'
	);

	private static $url_handlers = array(
		'location/$Location!' => 'location',
		'category/$Category!' => 'category',
	);

	protected $events;

	public function index()
	{
		$this->events = $this->getEvents('upcoming');

		return $this->render();
	}

	public function past()
	{
		$this->events = $this->getEvents('past');

		return $this->render();
	}

	public function location()
	{
		$tag = $this->getCurrentLocation();
		if ($tag) {
			$this->events = $tag->Events();

			return $this->render();
		}

		return $this->httpError(404, "Not Found");
	}

	public function category()
	{
		$category = $this->getCurrentCategory();
		if ($category) {
			$this->events = $category->Events();

			return $this->render();
		}

		return $this->httpError(404, "Not Found");
	}

	public function rss()
	{
		$rss = new RSSFeed($this->getEvents(), $this->Link(), $this->MetaTitle, $this->MetaDescription);
		$this->extend('updateRss', $rss);

		return $rss->outputToBrowser();
	}

	public function getCurrentLocation()
	{
		$location = $this->request->param("Location");
		if ($location) {
			return $this->dataRecord->EventLocations()
				->filter("URLSegment", $location)
				->first();
		}

		return null;
	}

	public function getCurrentCategory()
	{
		$category = $this->request->param("Category");
		if ($category) {
			return $this->dataRecord->EventCategories()
				->filter("URLSegment", $category)
				->first();
		}

		return null;
	}

	/**
	 * Returns a link to the RSS feed.
	 *
	 * @return string URL
	 **/
	public function getRSSLink()
	{
		return $this->Link("rss");
	}

	/**
	 * Returns a list of paginated blog posts based on the blogPost dataList
	 *
	 * @return PaginatedList
	 **/
	public function PaginatedList()
	{
		$events = new PaginatedList($this->events);

		// If pagination is set to '0' then no pagination will be shown.
		if ($this->EventsPerPage > 0) $events->setPageLength($this->EventsPerPage);
		else $events->setPageLength(999);

		$start = $this->request->getVar($events->getPaginationGetVar());
		$events->setPageStart($start);

		return $events;
	}
}