<?php

/**
 * Class FoxcommerceProduct
 *
 * @property float  $ItemPrice
 * @property string $InternalItemID Varchar
 * @property int    $MaximumPurchasableQuantity
 * @property int    $ImageID
 * @property int    $ShippingWarehouseID
 * @property int    $ShippingClassID
 * @method Image Image()
 * @method FoxcommerceShippingWarehouse ShippingWarehouse()
 * @method FoxcommerceShippingClass ShippingClass()
 * @method DataList|FoxcommerceProductAttributeValueLink[] AttributeValues()
 * @method ManyManyList|FoxcommerceProductAttribute[] Attributes()
 * @method ManyManyList|ProductImage[] Images()
 * @mixin FoxcommerceShippingWarehouseProductExtension
 * @mixin FoxcommerceProduct_TaxExtension
 * @mixin FoxcommerceProduct_MultipleImagesExtension
 * @mixin FoxcommerceProduct_ShippingClassExtension
 * @mixin MaximumQuantityRestrictedProductExtension
 */
class FoxcommerceProduct extends Page
{

    protected static $singular_name = 'Product';
    protected static $plural_name   = 'Products';

    protected static $icon = 'foxcommerce/images/product-icon.png';

    private static $db = [
        'ItemPrice'      => 'FixedDecimal',
        'InternalItemID' => 'Varchar'
    ];

    private static $has_one = [
        'Image' => 'Image'
    ];

    private static $has_many = [
        'AttributeValues' => 'FoxcommerceProductAttributeValueLink'
    ];

    private static $many_many = [
        'Attributes' => 'FoxcommerceProductAttribute'
    ];
    
    protected static $summary_fields = [
        'ID'             => 'Page ID',
        'InternalItemID' => 'Product Code',
        'Title'          => 'Product Name'
    ];

    private static $cart_summary = [
        'ID',
        'InternalItemID',
        'LastEdited'              => 'RevisionDate',
        'Title',
        'getMenuTitle'            => 'MenuTitle',
        'getPriceForCalculations' => 'BasePrice',
        'getExchangedPrice'       => 'ExchangedPrice',
        'getDisplayPrice'         => 'DisplayPrice',
        'getDisplayItemPrice'     => 'DisplayItemPrice',
        'Link',
        'getAttributesSummary'    => 'Attributes',
    ];

    public function getCMSFields()
    {

        $this->beforeUpdateCMSFields(function (FieldList $fields) {
            $categories = FoxcommerceProductCategory::get()->map()->toArray();

            $fields->addFieldsToTab('Root.ProductDetails', [
                TextField::create('InternalItemID', 'Product code'),
                TextField::create('ItemPrice', 'Price')->setDescription(
                    _t('FoxcommerceProduct.ItemPriceMessage',
                        'Enter the item price in your base currency ({base_currency})',
                        ['base_currency' => FoxcommerceCurrency::baseCurrency()->Code]
                    )
                )
            ]);

            $fields->addFieldsToTab('Root.Main', [
                DropdownField::create('ParentID', 'Primary category', $categories),
                UploadField::create('Image', 'Product image')
                    ->setAllowedFileCategories('image')
                    ->setFolderName('Images/Products')
            ], 'Content');

            $attributes = FoxcommerceProductAttribute::get();
            if ($attributes->count()) {

                $attributeChangeMessage      = _t(
                    'FoxcommerceProduct.AttributeChangeMessage',
                    "When you add an attribute from a product you'll need to save before you can set it's value"
                );
                $attributeChangeMessageField = LiteralField::create(
                    "AttributeChangeMessageHeader",
                    "<p class=\"message warning\">{$attributeChangeMessage}</p>"
                );

                $attributeOptions = ListboxField::create('Attributes', 'Product Attributes',
                    $attributes->map()->toArray());
                $attributeOptions->setMultiple(true);

                $fields->addFieldsToTab('Root.ProductAttributes', [
                    $attributeChangeMessageField,
                    $attributeOptions
                ]);

                $attributeValueMap = $this->AttributeValues()->map('AttributeID', 'AttributeValueID')->toArray();

                /** @var FoxcommerceProductAttribute $attribute */
                foreach ($this->Attributes() as $attribute) {
                    $fieldName   = "ProductAttribute[{$attribute->ID}]";
                    $fieldTitle  = $attribute->Title;
                    $fieldSource = $attribute->Values()->map()->toArray();
                    $fieldValue  = isset($attributeValueMap[$attribute->ID]) ? $attributeValueMap[$attribute->ID] : null;
                    $fields->addFieldToTab('Root.ProductAttributes',
                        DropdownField::create($fieldName, $fieldTitle, $fieldSource, $fieldValue)
                            ->setEmptyString('(none)'));
                }
            }
        });

        $fields = parent::getCMSFields();


        return $fields;
    }

    public function getPriceForCalculations($type = 'altered')
    {
        $itemPrice = $this->ItemPrice;
        $this->extend('updatePriceForCalculations', $itemPrice, $type);

        return $itemPrice;
    }

    public function getExchangedPrice($useOriginalItemPrice = false)
    {
        $price = $this->getPriceForCalculations($useOriginalItemPrice ? 'original' : 'altered');
        $order = FoxcommerceOrder::current();
        return BCMath::multiply($price, $order->Country()->Currency()->ExchangeRate);
    }

    public function getSummaryTitle()
    {
        if ($this->InternalItemID) {
            return "{$this->Title} ({$this->InternalItemID})";
        } else {
            return $this->Title;
        }
    }

    public function getIsEnquiryOnly()
    {
        return !floatval($this->getPriceForCalculations());
    }

    public function getDisplayItemPrice()
    {

        $order          = FoxcommerceOrder::current();
        $exchangedPrice = $this->getExchangedPrice(true);

        $extended = $this->extend('getDisplayItemPrice', $exchangedPrice, $order);
        if(!empty($extended)) {
            return $extended[0];
        }
        $oldLocale      = setlocale(LC_MONETARY, $order->current()->Country()->LocaleCode);
        $formattedPrice = money_format('%.2n', $exchangedPrice);
        setlocale(LC_MONETARY, $oldLocale);
        return $formattedPrice;
    }

    public function getDisplayPrice()
    {

        $order          = FoxcommerceOrder::current();
        $exchangedPrice = $this->getExchangedPrice();
        $extended = $this->extend('getDisplayPrice', $exchangedPrice, $order);
        if(!empty($extended)) {
            return $extended[0];
        }

        $oldLocale      = setlocale(LC_MONETARY, $order->current()->Country()->LocaleCode);
        $formattedPrice = money_format('%.2n', $exchangedPrice);
        setlocale(LC_MONETARY, $oldLocale);
        return $formattedPrice;
    }

    protected function onBeforeWrite()
    {
        parent::onBeforeWrite();
        foreach ($this->record as $key => $value) {
            if (preg_match('/ProductAttribute\[(\d+)\]/', $key, $matches)) {
                /** @var FoxcommerceProductAttributeValueLink $attributeLink */
                $attributeLink = $this->AttributeValues()->filter(['AttributeID' => $matches[1]])->first();
                if ($attributeLink && $attributeLink->AttributeValueID != $value) {
                    $attributeLink->AttributeValueID = $value;
                    $attributeLink->write();
                } elseif (!$attributeLink) {
                    /** @var FoxcommerceProductAttributeValueLink $attributeLink */
                    $attributeLink                   = FoxcommerceProductAttributeValueLink::create();
                    $attributeLink->ProductID        = $this->ID;
                    $attributeLink->AttributeID      = $matches[1];
                    $attributeLink->AttributeValueID = $value;
                    $attributeLink->write();
                }
            }
        }
    }

    public function getAttributesSummary()
    {
        $attributes = [];

        foreach ($this->AttributeValues() as $attr) {
            $attrID  = $attr->AttributeID;
            $valueID = $attr->AttributeValueID;

            if (!isset($attributes[$attrID])) {
                $attributes[$attrID] = [
                    'Attribute' => FoxcommerceHelper::build_cart_summary($attr->Attribute()),
                    'Values'    => []
                ];
            }

            $attributes[$attrID]['Values'][$valueID] = FoxcommerceHelper::build_cart_summary($attr->AttributeValue());
        }

        return $attributes;

    }

}

/**
 * Class FoxcommerceProduct_Controller
 *
 * @property FoxcommerceProduct dataRecord
 * @method FoxcommerceProduct data()
 * @mixin FoxcommerceProduct dataRecord
 */
class FoxcommerceProduct_Controller extends Page_Controller
{
    public function init()
    {
        parent::init();
    }

    public function CartInitialState()
    {
        $state                        = FoxcommerceHelper::initial_state();
        $state['products'][]          = FoxcommerceHelper::build_cart_summary($this->data());
        $state['productsDisplayed'][] = $this->data()->ID;

        return json_encode($state);
    }
}
