<?php

/**
 * Class FoxcommerceCheckoutPage
 *
 */
class FoxcommerceCheckoutPage extends Page
{

    protected static $singular_name = 'Checkout Page';
    protected static $plural_name   = 'Checkout Pages';

    protected static $icon = 'foxcommerce/images/basket-icon.png';


    function requireDefaultRecords()
    {

        if (self::config()->get('skip_creation')) {
            DB::alteration_message(get_class($this) . ' creation disabled by configuration.', 'error');
            return;
        }
        if (!FoxcommerceCheckoutPage::get_one('FoxcommerceCheckoutPage')) {
            $page             = new FoxcommerceCheckoutPage();
            $page->Title      = 'Checkout';
            $page->Content    = '';
            $page->URLSegment = 'checkout';
            $page->Status     = 'Published';
            $page->Sort       = 1;
            $page->write();
            $page->publish('Stage', 'Live');
            $page->flushCache();
            DB::alteration_message('Checkout page created', 'created');
        }
    }

    public function getCMSFields()
    {

        $this->beforeUpdateCMSFields(function (FieldList $fields) {

        });

        $fields = parent::getCMSFields();
        $fields->removeByName('Content');


        return $fields;
    }

    public function CheckoutSteps()
    {
        return Config::inst()->get('Checkout', 'steps');
    }

}

/**
 * Class FoxcommerceCheckoutPage_Controller
 *
 * @property FoxcommerceCheckoutPage dataRecord
 * @method FoxcommerceCheckoutPage data()
 * @mixin FoxcommerceCheckoutPage dataRecord
 */
class FoxcommerceCheckoutPage_Controller extends Page_Controller
{

    /** @var  FoxcommerceOrder */
    protected $currentOrder;

    protected static $allowed_actions = [
        'handle',
        'account',
        'proceedAsGuest',
        'CheckoutForm',
        'LoginForm'
    ];

    private static $url_handlers = [
        'CheckoutForm'       => 'CheckoutForm',
        'LoginForm'          => 'LoginForm',
        'account'            => 'account',
        'guest'              => 'proceedAsGuest',
    ];

    /**
     * Standard SilverStripe init()
     *
     * @return SS_HTTPResponse
     */
    public function init()
    {
        parent::init();

        //Add the catch all handler now so extensions have already added their handlers
        $handlerClass                           = ($this->class) ? $this->class : get_class($this);
        $existingHandlers                       = Config::inst()->get($handlerClass, 'url_handlers', Config::UNINHERITED);
        $existingHandlers['$Action//$ID/$Name'] = 'handle';
        Config::inst()->update($handlerClass, 'url_handlers', $existingHandlers);

        $urlHandlers = Config::inst()->get($handlerClass, 'url_handlers', Config::UNINHERITED);

        $this->currentOrder = FoxcommerceOrder::current();

        //if the user has no items in their cart redirect them to the cart instead
        //which displays a nice "you have no items in your cart" message
        if (!FoxcommerceCheckoutPage::config()->get('enable_on_empty_cart') && !$this->currentOrder->Items()->count()) {
            return $this->redirect(FoxcommerceCartPage::get_one('FoxcommerceCartPage')->Link());
        }
    }

    public function account(SS_HTTPRequest $request)
    {
        $order = FoxcommerceOrder::current();
        if ($order->IsGuestCart) {
            $order->IsGuestCart = false;
            $order->write();
        }

        return $this->customise([
            'Title' => 'New or returning customer?',
        ]);
    }

    public function proceedAsGuest(SS_HTTPRequest $request)
    {
        $order = FoxcommerceOrder::current();
        if (!$order->MemberID) {
            $order->IsGuestCart = true;
            $order->write();
        }

        return $this->redirect($this->Link('your-details'));
    }

    public function LoginForm()
    {
        return new FoxcommerceLoginForm($this, "LoginForm");
    }

    public function getDefaultAccountLink()
    {
        return $this->Link('account');
    }

    /**
     * Handles the flow of the entire checkout process, everything comes through here
     *
     * @param SS_HTTPRequest $request
     * @return SS_HTTPResponse|ViewableData_Customised
     */
    public function handle(SS_HTTPRequest $request)
    {
        $process = new CheckoutProcess($request, $this->currentOrder);

        //If we don't have an action (so we're just on /checkout for example) then redirect to the first step
        if (!$request->param('Action')) {
            return $this->redirect($this->Link(CheckoutProcess::stepByIndex(0)['url_segment']));
        }

        //If the current *requested* step is not valid redirect to the last valid step
        if (!$process->currentStepIsValid()) {
            return $this->redirect($this->Link($process->currentStep()['url_segment']));
        }

        //All the data you could possible want
        $stepClass = $process->currentStep()['classname'];
        /** @var CheckoutStep $stepInstance */
        $stepInstance = new $stepClass($this->currentOrder);

        $isMemberOrGuest = $this->currentOrder->MemberID || $this->currentOrder->IsGuestCart;
        if (!$stepInstance->accessibleByGuest() && !$isMemberOrGuest) {
            $accountLink = $this->getDefaultAccountLink();
            if ($accountLinkMethod = FoxcommerceCheckoutPage::config()->get('account_link_method')) {
                $accountLink = $this->data()->{$accountLinkMethod}();
            }
            return $this->redirect($accountLink);
        }


        $initializationResponse = $stepInstance->init();
        if ($initializationResponse instanceof SS_HTTPResponse) {
            return $initializationResponse;
        } elseif (in_array($initializationResponse, ['advance', 'rewind'])) {
            //The next/previous steps may not need us to redirect to them to process
            //so we ask them to short circuit and if so we advance to the next step
            $process->moveStep($initializationResponse, new CheckoutForm($process, $stepInstance, $this));
            $stepSegment = $process->currentStep()['url_segment'];
            return $this->redirect($this->Link($stepSegment));
        }

        $viewData = [
            'Title'        => $this->data()->getTitle() . ' - ' . $process->currentStep()['title'],
            'Process'      => $process,
            'StepList'     => $this->buildStepList($process),
            'Step'         => $stepInstance,
            'CurrentStep'  => $process->currentStep(),
            'NextStep'     => $process->nextStep(),
            'PreviousStep' => $process->previousStep()
        ];

        //Allow developers to modify the view data
        $this->extend('updateViewData', $viewData);

        return $this->customise($viewData);

    }

    /**
     * Builds the form for the checkout process passing through the known step information
     *
     * @return CheckoutForm
     */
    public function CheckoutForm()
    {
        $step      = $this->request->postVar('Step') ?: null;
        $process   = new CheckoutProcess($this->request, $this->currentOrder, $step);
        $stepClass = $process->currentStep()['classname'];
        $step      = new $stepClass($this->currentOrder);
        return new CheckoutForm($process, $step, $this);
    }

    /**
     * Builds the steplist for injection into the template as a variable
     *
     * @param CheckoutProcess $process
     * @return ArrayList
     */
    protected function buildStepList(CheckoutProcess $process)
    {
        $stepList = new ArrayList();
        foreach (CheckoutProcess::stepList() as $step => $stepData) {
            $isCurrent = $process->currentStep()['classname'] === $stepData['classname'];
            if ($isCurrent) $currentStepFound = false;
            $loopedStepClass = $stepData['classname'];
            /** @var CheckoutStep $loopedStep */
            $loopedStep = new $loopedStepClass($this->currentOrder);
            $stepList->push([
                'Title'      => $stepData['title'],
                'IsLinkable' => !$this->currentOrder->getIsPaid(),
                'Link'       => $this->Link($stepData['url_segment']),
                'IsCurrent'  => $isCurrent,
                'IsVisible'  => $loopedStep->visibleInStepList(),
                'IsNext'     => $process->nextStep()['classname'] === $stepData['classname'],
                'IsPrevious' => $process->previousStep()['classname'] === $stepData['classname'],
                'IsPast'     => !isset($currentStepFound)
            ]);
        }

        return $stepList;
    }

}
