<?php

/**
 * Class FoxcommerceOrderItem
 *
 * @property int $ProductVersionID
 * @property int $Quantity
 * @property int $OrderID
 * @property int $ProductID
 * @method FoxcommerceOrder Order()
 * @method DataList|FoxcommerceOrderItemAttributeValueLink[] AttributeValues()
 * @mixin TestingOrderItemExtension
 */
class FoxcommerceOrderItem extends DataObject
{

    protected static $db = [
        'ProductVersionID' => 'Int',
        'Quantity'         => 'Int'
    ];

    protected static $has_one = [
        'Order'   => 'FoxcommerceOrder',
        'Product' => 'FoxcommerceProduct'
    ];

    private static $has_many = [
        'AttributeValues' => 'FoxcommerceOrderItemAttributeValueLink'
    ];

    private static $summary_fields = [
        'ProductCode'  => 'Product Code',
        'ProductTitle' => 'Product Title',
        'Quantity'     => 'Quantity',
    ];

    private static $cart_summary = [
        'ID',
        'Created',
        'LastEdited',
        'ProductVersionID',
        'ProductID',
        'Quantity',
        'getLineTotal'        => 'BaseLineTotal',
        'getDisplayLineTotal' => 'DisplayLineTotal',
    ];

    private static $cart_summary_relations = [
        'Product',
        'AttributeValues' => 'Attributes'
    ];

    public function getCMSFields()
    {
        $product = $this->Product();

        $productEditLink = '<a href="' . $product->CMSEditLink() . '">' . $product->MenuTitle . '</a>';

        $productTitleField = ReadonlyField::create(
            'ProductTitle', 'Product', $productEditLink
        );

        $productTitleField->dontEscape = true;
        $productTitleField->setValue($productEditLink);

        $fields = new FieldList([
            $productTitleField
        ]);

        return $fields;
    }

    /**
     * @return FoxcommerceProduct
     */
    function Product()
    {
        return Versioned::get_version('FoxcommerceProduct', $this->ProductID, $this->ProductVersionID);
    }

    public function updateQuantity($newQuantity)
    {

        $oldQuantity    = $this->Quantity;
        $this->Quantity = $newQuantity;

        $beforeUpdateQuantity = $this->extend('beforeUpdateQuantity', $oldQuantity, $newQuantity);
        if (!empty($beforeUpdateQuantity)) {
            return count($beforeUpdateQuantity) === 1 ? array_pop($beforeUpdateQuantity) : $beforeUpdateQuantity;
        }

        if (!$newQuantity) {
            $this->delete();
        } else {
            $this->write();
        }

        $this->extend('afterUpdateQuantity', $this, $oldQuantity, $newQuantity);

        return $this;
    }

    public function getLineTotal()
    {
        return bcmul($this->Product()->getPriceForCalculations(), (string)$this->Quantity);
    }

    public function getProductTitle()
    {
        return $this->Product()->MenuTitle;
    }

    public function getProductCode()
    {
        return $this->Product()->InternalItemID;
    }

    public function getLineTotalAsBase()
    {
        return $this->getLineTotal() . ' ' . FoxcommerceCurrency::base()->Code;
    }

    public function getDisplayLineTotal()
    {
        $order = FoxcommerceOrder::current();
        /** @var float $exchangedPrice */
        $exchangedPrice = (bcmul($this->getLineTotal(), $order->Country()->Currency()->ExchangeRate));

        $oldLocale      = setlocale(LC_MONETARY, $order->current()->Country()->LocaleCode);
        $formattedPrice = money_format('%.2n', $exchangedPrice);
        setlocale(LC_MONETARY, $oldLocale);
        return $formattedPrice;
    }

    public function Attribute($type)
    {
        return $this->AttributeValues()->filter('Attribute.ClassName', $type)->first();
    }

    protected function onBeforeWrite()
    {
        parent::onBeforeWrite();

        if (!$this->OrderID) {
            throw new Exception("Attempting to write FoxcommerceOrderItem when no Order has been associated!");
        }
    }

    public function canCreate($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canView($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canEdit($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canDelete($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }


}