<?php

/**
 * Class FoxcommerceOrder
 *
 * @property boolean $Closed
 * @property boolean $IsGuestCart
 * @property string  $SessionID
 * @property float   $Subtotal
 * @property float   $NonModifiedSubtotal
 * @property float   $Total
 * @property float   $NonModifiedTotal
 * @property string  $CustomerFirstName
 * @property string  $CustomerSurname
 * @property string  $CustomerEmail
 * @property string  $CustomerTemporaryPassword
 * @property string  $ShippingAddressLine1
 * @property string  $ShippingAddressLine2
 * @property string  $ShippingAddressSuburb
 * @property string  $ShippingAddressRegion
 * @property string  $ShippingAddressPostcode
 * @property string  $ShippingAddressCountry
 * @property string  $BillingAddressLine1
 * @property string  $BillingAddressLine2
 * @property string  $BillingAddressSuburb
 * @property string  $BillingAddressRegion
 * @property string  $BillingAddressPostcode
 * @property string  $BillingAddressCountry
 * @property string  $PaymentGateway
 * @property int     $PageID
 * @property int     $CountryID
 * @property int     $MemberID
 * @property int     $BillingAddressForMemberID
 * @property int     $ShippingAddressForMemberID
 * @method Page Page()
 * @method Member Member()
 * @method FoxcommerceAddress BillingAddressForMember()
 * @method FoxcommerceAddress ShippingAddressForMember()
 * @method DataList|FoxcommerceAppliedOrderModifier[] Modifiers()
 * @method DataList|FoxcommerceOrderItem[] Items()
 * @method DataList|Payment[] Payments()
 * @mixin PayableUIExtension
 * @mixin Foxcommerce_OrderCompleteEmailsExtension
 * @mixin PayableUIExtension
 * @mixin FoxcommerceOrderCheckoutFieldsExtension
 */
class FoxcommerceOrder extends DataObject
{

    protected static $db = [
        'Closed'              => 'Boolean',
        'IsGuestCart'         => 'Boolean',
        'SessionID'           => 'Varchar(255)',
        'Subtotal'            => 'FixedDecimal',
        'Total'               => 'FixedDecimal',
        'NonModifiedSubtotal' => 'FixedDecimal',
        'NonModifiedTotal'    => 'FixedDecimal',

        'CustomerFirstName' => 'Varchar(255)',
        'CustomerSurname'   => 'Varchar(255)',
        'CustomerEmail'     => 'Varchar(255)',
    ];

    protected static $export_columns = [
        'Identifier'        => 'ID',
        'CustomerFirstName' => 'First Name',
        'CustomerSurname'   => 'Last Name',
        'CustomerEmail'     => 'Email',
        'Total'             => 'Total',
        'TotalPaid'         => 'Paid',
        'ItemQuantity'      => 'Items',
        'Closed.Nice'       => 'Closed',
        'Created'           => 'Created',
        'LastEdited'        => 'Last Edited'
    ];

    protected static $has_many = [
        'Modifiers' => 'FoxcommerceAppliedOrderModifier',
        'Items'     => 'FoxcommerceOrderItem',
    ];

    protected static $has_one = [
        'Page'    => 'Page',
        'Country' => 'FoxcommerceCountry',
        'Member'  => 'Member'
    ];

    protected static $extensions = [
        'FoxcommerceOrderCheckoutFieldsExtension'
    ];

    protected static $defaults = [
        'Closed'      => false,
        'IsGuestCart' => false,
    ];

    protected static $summary_fields = [
        'Identifier'        => 'ID',
        'CustomerFirstName' => 'First Name',
        'CustomerSurname'   => 'Last Name',
        'Total'             => 'Total',
        'TotalPaid'         => 'Paid',
        'ItemQuantity'      => 'Items',
        'Closed.Nice'       => 'Closed',
        'Created'           => 'Created',
        'LastEdited'        => 'Last Edited'
    ];

    /** @var  \FoxcommerceOrder */
    protected static $current;


    /**
     * @param bool $forceFetch
     * @return \FoxcommerceOrder
     */
    public static function current($forceFetch = false)
    {

        if ($forceFetch) {
            self::$current = null;
        }

        //Try and match the order by the session ID
        if (!self::$current && Session::get('order-guid')) {
            self::$current = self::get_one(__CLASS__, ['Closed' => false, 'SessionID' => Session::get('order-guid')]);

            //Make sure the session ID matches the logged in user if we have one..
            if (self::$current && self::$current->MemberID && self::$current->MemberID !== Member::currentUserID()) {
                self::$current = null;
                Session::clear('order-guid');
            }
        }

        //If that didn't work and we're logged in try and find our last edited order
        if (!self::$current && Member::currentUserID()) {
            self::$current = self::get_one(__CLASS__, ['Closed' => false, 'MemberID' => Member::currentUserID()], true, 'LastEdited DESC');

            //If we've found it we'll update it's session ID so it's easier to grab later
            if (self::$current) {
                Session::set('order-guid', FoxcommerceHelper::guid());
                self::$current->SessionID = Session::get('order-guid');
                self::$current->write(false, false, true);
            }
        }

        //They haven't got an active order, or they're not logged in so we can't find it. That's a problem.
        if (!self::$current) {
            Session::set('order-guid', FoxcommerceHelper::guid());
            self::$current = new self(['SessionID' => Session::get('order-guid')]);
        }

        return self::$current;
    }

    public static function setCurrent(FoxcommerceOrder $order)
    {
        self::$current = $order;

        return $order;
    }

    public function close()
    {
        $order         = self::current();
        self::$current = null;
        if ($order->isInDB()) {
            $order->SessionID = null;
            $order->Closed    = true;
            //force recalculate totals
            $order->write(false, false, true);
        }

    }

    public function Country()
    {
        if ($this->CountryID) {
            return FoxcommerceCountry::get_by_id('FoxcommerceCountry', $this->CountryID);
        }

        return FoxcommerceConfig::current()->FallbackCountry();
    }

    public function paymentFields()
    {
        $data = [
            'transactionId'    => uniqid("ID"),
            'firstName'        => '',
            'lastName'         => '',
            'email'            => '',
            'company'          => '',
            'billingAddress1'  => '',
            'billingAddress2'  => '',
            'billingCity'      => '',
            'billingPostcode'  => '',
            'billingState'     => '',
            'billingCountry'   => '',
            'billingPhone'     => '',
            'shippingAddress1' => '',
            'shippingAddress2' => '',
            'shippingCity'     => '',
            'shippingPostcode' => '',
            'shippingState'    => '',
            'shippingCountry'  => '',
            'shippingPhone'    => ''
        ];
        foreach (self::config()->get('payment_fields') as $paymentKey => $orderKey) {
            $data[$paymentKey] = $this->relField($orderKey);
        }

        return $data;
    }

    protected function onBeforeWrite()
    {
        if (!$this->CountryID) {
            $this->CountryID = FoxcommerceConfig::current()->FallbackCountryID;
        }

        if (Member::currentUserID() && !$this->MemberID) {
            $this->MemberID = Member::currentUserID();
        }

        parent::onBeforeWrite();
    }

    public function ItemQuantity()
    {
        $items = "0";
        foreach ($this->Items() as $orderItem) {
            $items = BCMath::add($items, (string)$orderItem->Quantity);
        }

        return $items;
    }

    public function DisplayItemQuantity()
    {
        return number_format($this->ItemQuantity());
    }

    public function getSubTotalModifiersForDisplay()
    {
        $list = new ArrayList();
        $this->Modifiers()->filter(['Modifier.ApplicationStage' => 'Subtotal'])
            ->each(function (FoxcommerceAppliedOrderModifier $m) use (&$list) {
                if (!$m->getShouldDisplay()) return;
                $list->push(new ArrayData([
                    'Title'        => $m->Modifier()->getTitle(),
                    'DisplayValue' => $m->getDisplayAmount(),
                    'Description'  => $m->Modifier()->getDescription()
                ]));
            });

        return $list;
    }

    public function getTotalModifiersForDisplay()
    {
        $list = new ArrayList();
        $this->Modifiers()->filter(['Modifier.ApplicationStage' => 'Total'])
            ->each(function (FoxcommerceAppliedOrderModifier $m) use (&$list) {
                if (!$m->getShouldDisplay()) return;
                $list->push(new ArrayData([
                    'Title'        => $m->Modifier()->getTitle(),
                    'DisplayValue' => $m->getDisplayAmount(),
                    'Description'  => $m->Modifier()->getDescription()
                ]));
            });

        return $list;
    }

    public function updateSubTotal()
    {
        $subtotal = "0";
        foreach ($this->Items() as $orderItem) {
            $subtotal = BCMath::add($subtotal, $orderItem->getLineTotal());
        }

        $this->NonModifiedSubtotal = $subtotal;

        /** @var FoxcommerceAppliedOrderModifier[] $appliedModifiers */
        $appliedModifiers = $this->Modifiers()->filter(['Modifier.ApplicationStage' => 'Subtotal']);
        foreach ($appliedModifiers as $appliedModifier) {
            $subtotal = BCMath::add($subtotal, $appliedModifier->Amount);
        }

        $this->Subtotal = $subtotal;
    }

    public function updateTotal()
    {

        $total = $this->Subtotal;

        $this->NonModifiedTotal = $this->Subtotal;

        /** @var FoxcommerceAppliedOrderModifier[] $appliedModifiers */
        $appliedModifiers = $this->Modifiers()->filter(['Modifier.ApplicationStage' => 'Total']);
        foreach ($appliedModifiers as $appliedModifier) {
            $total = BCMath::add($total, $appliedModifier->Amount);
        }

        $this->Total = $total;
    }


    public function getCMSFields()
    {
        $currencyCode = FoxcommerceCurrency::base()->Code;

        $fields = new FieldList([
            new TabSet("Root", $mainTab = new Tab("Main"))
        ]);
        $mainTab->setTitle(_t('SiteTree.TABMAIN', "Main"));

        $fields->addFieldsToTab('Root.Main', [
            ReadonlyField::create('OrderID', 'Order ID', ('#' . $this->getIdentifier())),
            ReadonlyField::create('OrderTotal', 'Total', ($this->Total ?: '0.00')),
            ReadonlyField::create('OrderPaid', 'Paid', ($this->TotalPaid() ?: '0.00')),
            ReadOnlyField::create('CustomerFirstName', 'First Name', $this->CustomerFirstName),
            ReadOnlyField::create('CustomerSurname', 'Surname', $this->CustomerSurname),
            ReadOnlyField::create('CustomerEmail', 'Email Address', $this->CustomerEmail),
            DatetimeField::create('Created')->performReadonlyTransformation(),
            DatetimeField::create('LastEdited')->performReadonlyTransformation(),
        ]);

        if ($this->hasField('CustomerPhone')) {
            $fields->insertAfter('CustomerEmail', ReadonlyField::create('CustomerPhone', 'Phone'));
        }

        $orderItemsConfig = new GridFieldConfig_Base(300);
        $orderItemsConfig->addComponent(new GridFieldButtonRow('before'));
        $orderItemsConfig->addComponent(new GridFieldExportButton('buttons-before-left'));
        $fields->addFieldsToTab('Root.OrderItems', [
            GridField::create('OrderItems', 'Order Items', $this->Items(), $orderItemsConfig)
        ]);

        foreach (['Shipping', 'Billing'] as $type) {
            $fields->addFieldsToTab("Root.Addresses.{$type}", [
                ReadonlyField::create("{$type}AddressLine1", 'Address Line 1'),
                ReadonlyField::create("{$type}AddressLine2", 'Address Line 2'),
                ReadonlyField::create("{$type}AddressSuburb", 'Suburb'),
                ReadonlyField::create("{$type}AddressRegion", 'Region'),
                ReadonlyField::create("{$type}AddressPostcode", 'Postcode'),
                CountryDropdownField::create("{$type}AddressCountry", 'Country')->performReadonlyTransformation()
            ]);
        }

        $this->extend('updateCMSFields', $fields);

        return $fields;
    }

    public function hasItem($filterData)
    {
        return !!$this->getItem($filterData);
    }

    /**
     * @param $filterData
     * @return FoxcommerceOrderItem
     */
    public function getItem($filterData)
    {
        return $this->Items()->filter($filterData)->first();
    }

    public function getIsBaseCurrency()
    {
        return $this->Country()->CurrencyID === FoxcommerceCurrency::base()->ID;
    }

    public function getBillingAndShippingAreTheSame()
    {
        $difference = 0;
        $difference += strcmp($this->BillingAddressLine1, $this->ShippingAddressLine1);
        $difference += strcmp($this->BillingAddressLine2, $this->ShippingAddressLine2);
        $difference += strcmp($this->BillingAddressSuburb, $this->ShippingAddressSuburb);
        $difference += strcmp($this->BillingAddressRegion, $this->ShippingAddressRegion);
        $difference += strcmp($this->BillingAddressPostcode, $this->ShippingAddressPostcode);
        $difference += strcmp($this->BillingAddressCountry, $this->ShippingAddressCountry);

        return $difference === 0;
    }

    public function getCustomerName()
    {
        return trim("{$this->CustomerFirstName} {$this->CustomerSurname}") ?: 'Not Set';
    }

    public function getIdentifier()
    {
        $memberNumber = $this->MemberID ?: 99;
        $orderNumber  = $this->ID ?: 99;
        $createdTime  = strtotime($this->Created);
        return join('', [
            range(0, 9)[($createdTime + $orderNumber) % 9] . str_pad($memberNumber, 4, '0', STR_PAD_LEFT),
            range('A', 'Z')[($createdTime + $memberNumber) % 26],
            range(0, 9)[(5 + $createdTime) % 9] . str_pad($orderNumber, 4, '0', STR_PAD_LEFT)
        ]);
    }

    /**
     * Attempts to reassociate a cart with the users new session
     * If the session already has an existing cart but no items, the cart will be deleted
     * If the session already has an existing cart that contains items false will be returned
     *
     * @param null $sessionID
     * @return $this|bool
     */
    public function reassociate($sessionID = null)
    {
        $sessionID = $sessionID ?: Session::get('order-guid');
        /** @var self $existingCart */
        $existingCart = self::get_one(__CLASS__, ['SessionID' => $sessionID]);


        if ($existingCart && $existingCart->Items()->count() === 0) {
            $existingCart->delete();
            $existingCart = false;
        }

        if (!$existingCart) {
            $this->SessionID = $sessionID;
            //force recalculate totals
            $this->write(false, false, true);
            self::$current = null;
            return $this;
        } else {
            return false;
        }
    }

    /**
     * Attempts to reassociate a cart with the users new session
     * If the session already has an existing cart but no items, the cart will be deleted
     * If the session already has an existing cart that contains items false will be returned
     *
     * @param null $sessionID
     * @return $this|bool
     */
    public function disassociate()
    {
        $sessionID       = $this->SessionID;
        $memberID        = $this->MemberID;
        $this->SessionID = null;
        //force recalculate totals
        $this->write(false, false, true);
        $order            = new static();
        $order->SessionID = $sessionID;
        $order->MemberID  = $memberID;
        $order->write();
        self::$current = $order;
        return $order;

    }

    public function applyAutomaticModifiers()
    {

        $appliedModifiers = $this->Modifiers()->column('ModifierClassName');

        /** @var FoxcommerceOrderModifier[] $automaticModifiers */
        $automaticModifiers = FoxcommerceOrderModifier::get(
            'FoxcommerceOrderModifier',
            ['ApplicationTrigger' => 'Automatic']
        );

        foreach ($automaticModifiers as $modifier) {
            if (!in_array($modifier->ClassName, $appliedModifiers)) {
                $this->applyModifier($modifier);
            }
        }
    }

    public function refreshAppliedModifiers()
    {
        $appliedModifiers = $this->Modifiers()->filter(['Modifier.ApplicationStage' => 'Subtotal']);
        /** @var FoxcommerceAppliedOrderModifier[] $appliedModifiers */
        $appliedSubtotalModifiers = ArrayList::create();
        foreach ($appliedModifiers as $appliedModifier) {
            $appliedModifier->Amount = $appliedModifier->Modifier()->calculateForOrder($this);
            if ($appliedModifier->Modifier()->ModifierType === 'Deduction') {
                $appliedModifier->Amount = BCMath::multiply($appliedModifier->Amount, '-1');
            }
            $appliedModifier->write();
            $appliedSubtotalModifiers->push($appliedModifier);
        }

        $this->updateSubTotal();
        /** @var FoxcommerceAppliedOrderModifier[] $appliedModifiers */
        $appliedModifiers = $this->Modifiers()->filter(['Modifier.ApplicationStage' => 'Total']);
        foreach ($appliedModifiers as $appliedModifier) {
            $appliedModifier->Amount = $appliedModifier->Modifier()->calculateForOrder(
                $this,
                $appliedSubtotalModifiers
            );
            if ($appliedModifier->Modifier()->ModifierType === 'Deduction') {
                $appliedModifier->Amount = BCMath::multiply($appliedModifier->Amount, '-1');
            }
            $appliedModifier->write();
        }

        $this->updateTotal();

    }

    public function applyModifier(FoxcommerceOrderModifier $modifier)
    {
        //We shouldn't be adding modifiers to orders that don't exist
        if (!$this->exists()) return false;

        //If the order already has this modifier we don't need to readd it.
        if ($this->hasAppliedModifierByType($modifier)) return false;

        //Create the new applied modifier - Note that it's an "AppliedOrderModifier" rather than the modifier itself
        //We do this because the actual OrderModifier is just a record of the values and rules for the OrderModifier
        $appliedModifier = new FoxcommerceAppliedOrderModifier([
            'OrderID'           => $this->ID,
            'ModifierID'        => $modifier->ID,
            'ModifierClassName' => $modifier->ClassName
        ]);

        $appliedModifier->write();
        //force recalculate totals
        $this->write(false, false, true);

        return $appliedModifier;
    }

    /**
     * @param FoxcommerceOrderModifier|string $type
     */
    public function hasAppliedModifierByType($type)
    {
        if (!is_string($type)) {
            $type = $type->ClassName;
        }
        /** @var FoxcommerceAppliedOrderModifier[] $appliedModifiers */
        $appliedModifiers = $this->Modifiers()->filter(['ModifierClassName' => $type]);
        return $appliedModifiers->exists();
    }

    /**
     * @param FoxcommerceOrderModifier|string $type
     */
    public function removeAppliedModifierByType($type)
    {
        if (!is_string($type)) {
            $type = $type->ClassName;
        }

        /** @var HasManyList|FoxcommerceAppliedOrderModifier[] $appliedModifiers */
        $appliedModifiers = $this->Modifiers()->filter(['ClassName' => $type]);
        if ($appliedModifiers->exists()) {
            foreach ($appliedModifiers as $appliedModifier) {
                $appliedModifier->delete();
            }
        }
    }

    public function getDisplayTotal()
    {
        $exchangedPrice = BCMath::multiply($this->Total, $this->Country()->Currency()->ExchangeRate);

        $oldLocale      = setlocale(LC_MONETARY, $this->current()->Country()->LocaleCode);
        $formattedPrice = money_format('%.2n', $exchangedPrice);
        setlocale(LC_MONETARY, $oldLocale);
        return $formattedPrice;
    }

    public function getNonModifiedDisplayTotal()
    {
        /** @var float $exchangedPrice */
        $exchangedPrice = BCMath::multiply($this->NonModifiedTotal, $this->Country()->Currency()->ExchangeRate);

        $oldLocale      = setlocale(LC_MONETARY, $this->current()->Country()->LocaleCode);
        $formattedPrice = money_format('%.2n', $exchangedPrice);
        setlocale(LC_MONETARY, $oldLocale);
        return $formattedPrice;
    }

    public function getIsCurrent()
    {
        return $this->ID === self::current()->ID;
    }

    public function getIsClosed()
    {
        return $this->Closed;
    }

    public function getDisplaySubtotal()
    {
        $exchangedPrice = BCMath::multiply($this->Subtotal, $this->Country()->Currency()->ExchangeRate);

        $oldLocale      = setlocale(LC_MONETARY, $this->current()->Country()->LocaleCode);
        $formattedPrice = money_format('%.2n', $exchangedPrice);
        setlocale(LC_MONETARY, $oldLocale);
        return $formattedPrice;
    }

    public function getNonModifiedDisplaySubtotal()
    {
        $exchangedPrice = BCMath::multiply($this->NonModifiedSubtotal, $this->Country()->Currency()->ExchangeRate);

        $oldLocale      = setlocale(LC_MONETARY, $this->current()->Country()->LocaleCode);
        $formattedPrice = money_format('%.2n', $exchangedPrice);
        setlocale(LC_MONETARY, $oldLocale);
        return $formattedPrice;
    }

    public function getIsPaid()
    {
        return $this->Payments()->exists() && BCMath::gte((string)$this->TotalPaid(), (string)$this->Total, 2);
    }

    public function toNestedArray()
    {
        $cart             = [];
        $cart['MetaData'] = array_merge(
            FoxcommerceHelper::array_except(
                FoxcommerceConfig::current()->toMap(),
                ['ClassName', 'ID', 'RecordClassName', 'LastEdited', 'Created']
            ),
            [
                'LastEdited' => $this->LastEdited,
                'Created'    => $this->Created,
                'Country'    => FoxcommerceHelper::build_cart_summary($this->Country())
            ]
        );

        $cart['Items'] = [];
        foreach ($this->Items() as $orderItem) {
            $cart['Items'][] = FoxcommerceHelper::build_cart_summary($orderItem);
        }

        $cart['Modifiers'] = ['SubtotalModifiers' => [], 'TotalModifiers' => []];

        $appliedModifiers = $this->Modifiers()->filter(['Modifier.ApplicationStage' => 'Subtotal']);
        foreach ($appliedModifiers as $appliedModifier) {
            $cart['Modifiers']['SubtotalModifiers'][] = FoxcommerceHelper::build_cart_summary($appliedModifier);
        }

        $appliedModifiers = $this->Modifiers()->filter(['Modifier.ApplicationStage' => 'Total']);
        foreach ($appliedModifiers as $appliedModifier) {
            $cart['Modifiers']['TotalModifiers'][] = FoxcommerceHelper::build_cart_summary($appliedModifier);
        }

        $cart['Totals'] = [
            'Total'                      => $this->Total,
            'Subtotal'                   => $this->Subtotal,
            'DisplayTotal'               => $this->getDisplayTotal(),
            'DisplaySubtotal'            => $this->getDisplaySubtotal(),
            'NonModifiedTotal'           => $this->NonModifiedTotal,
            'NonModifiedSubtotal'        => $this->NonModifiedSubtotal,
            'NonModifiedDisplayTotal'    => $this->getNonModifiedDisplayTotal(),
            'NonModifiedDisplaySubtotal' => $this->getNonModifiedDisplaySubtotal()
        ];


        $this->extend('toNestedArray', $cart);

        return $cart;
    }

    protected function onAfterSkippedWrite()
    {
        $this->calculateTotalsOnChange();
    }

    protected function onAfterWrite()
    {
        parent::onAfterWrite();

        $this->calculateTotalsOnChange();
    }

    protected function calculateTotalsOnChange()
    {

        $originalSubtotal = $this->Subtotal;
        $originalTotal    = $this->Total;

        $this->applyAutomaticModifiers();
        $this->refreshAppliedModifiers();

        //If these don't match it means our modifiers have adjusted the values
        if (!BCMath::eq($originalSubtotal, $this->Subtotal) || !BCMath::eq($originalTotal, $this->Total)) {
            //force recalculate totals
            $this->write(false, false, true);
        }
    }

    public function canCreate($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canView($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canEdit($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canDelete($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }


}