<?php

/**
 * Class FoxcommerceCurrency
 *
 * @property string $NameVarchar|
 * @property string $CodeVarchar|
 * @property string $SymbolVarchar|
 * @property float $ExchangeRateDecimal|
 * @property boolean $IsBaseCurrency
 * @property boolean $Enabled
 * @method DataList|FoxcommerceCountry[] Countries()
 */
class FoxcommerceCurrency extends DataObject
{

    protected static $singular_name = 'Currency';
    protected static $plural_name   = 'Currencies';

    protected static $db = [
        'Name'           => 'Varchar(255)',
        'Code'           => 'Varchar',
        'Symbol'         => 'Varchar',
        'ExchangeRate'   => 'FixedDecimal',
        'IsBaseCurrency' => 'Boolean',
        'Enabled'        => 'Boolean'
    ];

    private static $has_many = [
        'Countries' => 'FoxcommerceCountry'
    ];

    private static $defaults = [
        'Enabled' => true
    ];

    private static $summary_fields = [
        'Name'                => 'Name',
        'Code'                => 'Code',
        'Symbol'              => 'Symbol',
        'ExchangeRate'        => 'Exchange Rate',
        'IsBaseCurrency.Nice' => 'Base Currency',
        'Enabled.Nice'        => 'Enabled'
    ];

    private static $default_sort = 'Name ASC';

    private static $better_buttons_actions = [
        'doSetAsBaseCurrency'
    ];

    private static $cart_summary = [
        'Name' => 'Title',
        'Code',
        'Symbol',
        'ExchangeRate'
    ];

    protected static $currency_symbols = [
        'ALL' => 'Lek',
        'AFN' => '؋',
        'ARS' => '$',
        'AWG' => 'ƒ',
        'AUD' => '$',
        'AZN' => 'ман',
        'BSD' => '$',
        'BBD' => '$',
        'BYR' => 'p.',
        'BZD' => 'BZ$',
        'BMD' => '$',
        'BOB' => '$b',
        'BAM' => 'KM',
        'BWP' => 'P',
        'BGN' => 'лв',
        'BRL' => 'R$',
        'BND' => '$',
        'KHR' => '៛',
        'CAD' => '$',
        'KYD' => '$',
        'CLP' => '$',
        'CNY' => '¥',
        'COP' => '$',
        'CRC' => '₡',
        'HRK' => 'kn',
        'CUP' => '₱',
        'CZK' => 'Kč',
        'DKK' => 'kr',
        'DOP' => 'RD$',
        'XCD' => '$',
        'EGP' => '£',
        'SVC' => '$',
        'EUR' => '€',
        'FKP' => '£',
        'FJD' => '$',
        'GHS' => '¢',
        'GIP' => '£',
        'GTQ' => 'Q',
        'GGP' => '£',
        'GYD' => '$',
        'HNL' => 'L',
        'HKD' => '$',
        'HUF' => 'Ft',
        'ISK' => 'kr',
        'INR' => '',
        'IDR' => 'Rp',
        'IRR' => '﷼',
        'IMP' => '£',
        'ILS' => '₪',
        'JMD' => 'J$',
        'JPY' => '¥',
        'JEP' => '£',
        'KZT' => 'лв',
        'KPW' => '₩',
        'KRW' => '₩',
        'KGS' => 'лв',
        'LAK' => '₭',
        'LBP' => '£',
        'LRD' => '$',
        'MKD' => 'ден',
        'MYR' => 'RM',
        'MUR' => '₨',
        'MXN' => '$',
        'MNT' => '₮',
        'MZN' => 'MT',
        'NAD' => '$',
        'NPR' => '₨',
        'ANG' => 'ƒ',
        'NZD' => '$',
        'NIO' => 'C$',
        'NGN' => '₦',
        'NOK' => 'kr',
        'OMR' => '﷼',
        'PKR' => '₨',
        'PAB' => 'B/.',
        'PYG' => 'Gs',
        'PEN' => 'S/.',
        'PHP' => '₱',
        'PLN' => 'zł',
        'QAR' => '﷼',
        'RON' => 'lei',
        'RUB' => 'руб',
        'SHP' => '£',
        'SAR' => '﷼',
        'RSD' => 'Дин.',
        'SCR' => '₨',
        'SGD' => '$',
        'SBD' => '$',
        'SOS' => 'S',
        'ZAR' => 'R',
        'LKR' => '₨',
        'SEK' => 'kr',
        'CHF' => 'CHF',
        'SRD' => '$',
        'SYP' => '£',
        'TWD' => 'NT$',
        'THB' => '฿',
        'TTD' => 'TT$',
        'TRY' => '',
        'TVD' => '$',
        'UAH' => '₴',
        'GBP' => '£',
        'USD' => '$',
        'UYU' => '$U',
        'UZS' => 'лв',
        'VEF' => 'Bs',
        'VND' => '₫',
        'YER' => '﷼',
        'ZWD' => 'Z$',
    ];

    protected static $currency_codes = [
        'AED' => 'United Arab Emirates Dirham',
        'AFN' => 'Afghan Afghani',
        'ALL' => 'Albanian Lek',
        'AMD' => 'Armenian Dram',
        'ANG' => 'Netherlands Antillean Guilder',
        'AOA' => 'Angolan Kwanza',
        'ARS' => 'Argentine Peso',
        'AUD' => 'Australian Dollar',
        'AWG' => 'Aruban Florin',
        'AZN' => 'Azerbaijani Manat',
        'BAM' => 'Bosnia-Herzegovina Convertible Mark',
        'BBD' => 'Barbadian Dollar',
        'BDT' => 'Bangladeshi Taka',
        'BGN' => 'Bulgarian Lev',
        'BHD' => 'Bahraini Dinar',
        'BIF' => 'Burundian Franc',
        'BMD' => 'Bermudan Dollar',
        'BND' => 'Brunei Dollar',
        'BOB' => 'Bolivian Boliviano',
        'BRL' => 'Brazilian Real',
        'BSD' => 'Bahamian Dollar',
        'BTC' => 'Bitcoin',
        'BTN' => 'Bhutanese Ngultrum',
        'BWP' => 'Botswanan Pula',
        'BYR' => 'Belarusian Ruble',
        'BZD' => 'Belize Dollar',
        'CAD' => 'Canadian Dollar',
        'CDF' => 'Congolese Franc',
        'CHF' => 'Swiss Franc',
        'CLF' => 'Chilean Unit of Account',
        'CLP' => 'Chilean Peso',
        'CNH' => 'CNH',
        'CNY' => 'Chinese Yuan',
        'COP' => 'Colombian Peso',
        'CRC' => 'Costa Rican Colón',
        'CUP' => 'Cuban Peso',
        'CVE' => 'Cape Verdean Escudo',
        'CZK' => 'Czech Republic Koruna',
        'DEM' => 'German Mark',
        'DJF' => 'Djiboutian Franc',
        'DKK' => 'Danish Krone',
        'DOP' => 'Dominican Peso',
        'DZD' => 'Algerian Dinar',
        'EGP' => 'Egyptian Pound',
        'ERN' => 'Eritrean Nakfa',
        'ETB' => 'Ethiopian Birr',
        'EUR' => 'Euro',
        'FIM' => 'Finnish Markka',
        'FJD' => 'Fijian Dollar',
        'FKP' => 'Falkland Islands Pound',
        'FRF' => 'French Franc',
        'GBP' => 'British Pound',
        'GEL' => 'Georgian Lari',
        'GHS' => 'Ghanaian Cedi',
        'GIP' => 'Gibraltar Pound',
        'GMD' => 'Gambian Dalasi',
        'GNF' => 'Guinean Franc',
        'GTQ' => 'Guatemalan Quetzal',
        'GYD' => 'Guyanaese Dollar',
        'HKD' => 'Hong Kong Dollar',
        'HNL' => 'Honduran Lempira',
        'HRK' => 'Croatian Kuna',
        'HTG' => 'Haitian Gourde',
        'HUF' => 'Hungarian Forint',
        'IDR' => 'Indonesian Rupiah',
        'IEP' => 'Irish Pound',
        'ILS' => 'Israeli New Sheqel',
        'INR' => 'Indian Rupee',
        'IQD' => 'Iraqi Dinar',
        'IRR' => 'Iranian Rial',
        'ISK' => 'Icelandic Króna',
        'ITL' => 'Italian Lira',
        'JMD' => 'Jamaican Dollar',
        'JOD' => 'Jordanian Dinar',
        'JPY' => 'Japanese Yen',
        'KES' => 'Kenyan Shilling',
        'KGS' => 'Kyrgystani Som',
        'KHR' => 'Cambodian Riel',
        'KMF' => 'Comorian Franc',
        'KPW' => 'North Korean Won',
        'KRW' => 'South Korean Won',
        'KWD' => 'Kuwaiti Dinar',
        'KYD' => 'Cayman Islands Dollar',
        'KZT' => 'Kazakhstani Tenge',
        'LAK' => 'Laotian Kip',
        'LBP' => 'Lebanese Pound',
        'LKR' => 'Sri Lankan Rupee',
        'LRD' => 'Liberian Dollar',
        'LSL' => 'Lesotho Loti',
        'LTL' => 'Lithuanian Litas',
        'LVL' => 'Latvian Lats',
        'LYD' => 'Libyan Dinar',
        'MAD' => 'Moroccan Dirham',
        'MDL' => 'Moldovan Leu',
        'MGA' => 'Malagasy Ariary',
        'MKD' => 'Macedonian Denar',
        'MMK' => 'Myanmar Kyat',
        'MNT' => 'Mongolian Tugrik',
        'MOP' => 'Macanese Pataca',
        'MRO' => 'Mauritanian Ouguiya',
        'MUR' => 'Mauritian Rupee',
        'MVR' => 'Maldivian Rufiyaa',
        'MWK' => 'Malawian Kwacha',
        'MXN' => 'Mexican Peso',
        'MYR' => 'Malaysian Ringgit',
        'MZN' => 'Mozambican Metical',
        'NAD' => 'Namibian Dollar',
        'NGN' => 'Nigerian Naira',
        'NIO' => 'Nicaraguan Córdoba',
        'NOK' => 'Norwegian Krone',
        'NPR' => 'Nepalese Rupee',
        'NZD' => 'New Zealand Dollar',
        'OMR' => 'Omani Rial',
        'PAB' => 'Panamanian Balboa',
        'PEN' => 'Peruvian Nuevo Sol',
        'PGK' => 'Papua New Guinean Kina',
        'PHP' => 'Philippine Peso',
        'PKG' => 'PKG',
        'PKR' => 'Pakistani Rupee',
        'PLN' => 'Polish Zloty',
        'PYG' => 'Paraguayan Guarani',
        'QAR' => 'Qatari Rial',
        'RON' => 'Romanian Leu',
        'RSD' => 'Serbian Dinar',
        'RUB' => 'Russian Ruble',
        'RWF' => 'Rwandan Franc',
        'SAR' => 'Saudi Riyal',
        'SBD' => 'Solomon Islands Dollar',
        'SCR' => 'Seychellois Rupee',
        'SDG' => 'Sudanese Pound',
        'SEK' => 'Swedish Krona',
        'SGD' => 'Singapore Dollar',
        'SHP' => 'St. Helena Pound',
        'SKK' => 'Slovak Koruna',
        'SLL' => 'Sierra Leonean Leone',
        'SOS' => 'Somali Shilling',
        'SRD' => 'Surinamese Dollar',
        'STD' => 'São Tomé &amp; Príncipe Dobra',
        'SVC' => 'Salvadoran Colón',
        'SYP' => 'Syrian Pound',
        'SZL' => 'Swazi Lilangeni',
        'THB' => 'Thai Baht',
        'TJS' => 'Tajikistani Somoni',
        'TMT' => 'Turkmenistani Manat',
        'TND' => 'Tunisian Dinar',
        'TOP' => 'Tongan Paʻanga',
        'TRY' => 'Turkish Lira',
        'TTD' => 'Trinidad &amp; Tobago Dollar',
        'TWD' => 'New Taiwan Dollar',
        'TZS' => 'Tanzanian Shilling',
        'UAH' => 'Ukrainian Hryvnia',
        'UGX' => 'Ugandan Shilling',
        'USD' => 'US Dollar',
        'UYU' => 'Uruguayan Peso',
        'UZS' => 'Uzbekistani Som',
        'VEF' => 'Venezuelan Bolívar',
        'VND' => 'Vietnamese Dong',
        'VUV' => 'Vanuatu Vatu',
        'WST' => 'Samoan Tala',
        'XAF' => 'Central African CFA Franc',
        'XCD' => 'East Caribbean Dollar',
        'XDR' => 'Special Drawing Rights',
        'XOF' => 'West African CFA Franc',
        'XPF' => 'CFP Franc',
        'YER' => 'Yemeni Rial',
        'ZAR' => 'South African Rand',
        'ZMK' => 'Zambian Kwacha',
        'ZMW' => 'Zambian Kwacha',
        'ZWL' => 'Zimbabwean Dollar',
    ];

    /**
     * @return FoxcommerceCurrency
     */
    public static function baseCurrency()
    {
        return static::get_one(__CLASS__, ['IsBaseCurrency' => true]);
    }

    public function getCMSFields()
    {

        $baseCurrency   = self::baseCurrency();
        $exchangeAmount = $this->ExchangeRate;

        $exchangeRateField = TextField::create('ExchangeRate')->setPlaceholder('0.85');

        $fields = new FieldList([
            DropdownField::create('Code', 'Code', array_combine(
                array_keys($this->stat('currency_codes')),
                array_keys($this->stat('currency_codes'))
            ))
        ]);

        if (!$this->exists()) {
            if (!$baseCurrency) {
                $firstCurrencyMessage = _t(
                    'FoxcommerceCurrency.FirstCurrencyMessage',
                    "As this is the first currency setup for your shop it will be marked as the primary currency."
                );
                $fields->insertBefore('Code', LiteralField::create("FirstCurrencyMessageHeader",
                    "<p class=\"message warning\">{$firstCurrencyMessage}</p>")
                );
            } else {
                $fields->insertAfter('Code', $exchangeRateField);
            }

        } else {

            $fields->insertAfter('Code', TextField::create('Name', 'Name'));
            $fields->insertAfter('Name', TextField::create('Symbol', 'Symbol')->setPlaceholder('$'));

            if ($this->IsBaseCurrency) {
                $baseCurrencyMessage = _t(
                    'FoxcommerceCurrency.BaseCurrencyMessage',
                    "This currency is your base currency so some options are not able to be changed."
                );
                $fields->insertBefore('Code', LiteralField::create(
                    "BaseCurrencyMessageHeader",
                    "<p class=\"message warning\">{$baseCurrencyMessage}</p>")
                );
            } else {
                $exchangeRateMessage = _t(
                    'FoxcommerceCurrency.ExchangeRateMessage',
                    'Current Exchange: 1{base_code} = {exchange_rate}{exchange_code}', [
                    'base_code'     => $baseCurrency->Code,
                    'exchange_rate' => $exchangeAmount,
                    'exchange_code' => $this->Code
                ]);

                $fields->insertAfter('Symbol', $exchangeRateField->setDescription($exchangeRateMessage));

                $fields->insertAfter('ExchangeRate',
                    DropdownField::create('Enabled', 'Enabled', [1 => 'Enabled', 0 => 'Disabled']));

            }
        }

        $this->extend('updateCMSFields', $fields);

        return $fields;

    }

    public function onBeforeWrite()
    {

        parent::onBeforeWrite();

        $codes   = $this->stat('currency_codes');
        $symbols = $this->stat('currency_symbols');

        if (!$this->Name && isset($codes[$this->Code])) {
            $this->Name = $codes[$this->Code];
        }

        if (!$this->Symbol && isset($symbols[$this->Code])) {
            $this->Symbol = $symbols[$this->Code];
        }

        if (!self::baseCurrency()) {
            $this->IsBaseCurrency = true;
            $this->ExchangeRate   = 1;
            $this->Enabled        = true;
        }

        if (self::baseCurrency() && $this->IsBaseCurrency && $this->ID !== self::baseCurrency()->ID) {
            $baseCurrency                 = self::baseCurrency();
            $baseCurrency->IsBaseCurrency = false;
            $baseCurrency->write();
        }
    }

    public function doSetAsBaseCurrency(BetterButtonCustomAction $data, GridFieldDetailForm_ItemRequest $form)
    {
        $this->IsBaseCurrency = true;
        $this->ExchangeRate   = 1;
        $this->write();
    }

    public function getBetterButtonsActions()
    {
        /** @var FieldList $fields */
        $fields = parent::getBetterButtonsActions();
        if (!$this->exists()) {
            $fields->removeByName('action_doSaveAndQuit');
        } elseif (!$this->IsBaseCurrency) {

            /** @var BetterButtonCustomAction $baseCurrencyButton */
            $baseCurrencyButton = BetterButtonCustomAction::create('doSetAsBaseCurrency', 'Set as base currency');
            $baseCurrencyButton->setRedirectType(BetterButtonCustomAction::REFRESH)->setSuccessMessage('Set as base currency');
            $baseCurrencyButton->setAttribute('data-icon', 'chain--arrow');
            $fields->push($baseCurrencyButton);
        }


        return $fields;

    }

    /**
     * @return FoxcommerceCurrency
     */
    public static function base()
    {
        return static::get_one(__CLASS__, ['IsBaseCurrency' => true]);
    }

    public function canDelete($member = null)
    {
        if ($this->IsBaseCurrency) {
            return false;
        }
        return parent::canDelete($member);
    }

    public function canCreate($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canView($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canEdit($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

}