<?php

/**
 * Class FoxcommerceCountry
 *
 * @property string $Title
 * @property string $Code
 * @property string $LocaleCode
 * @property boolean $Enabled
 * @property int $CurrencyID
 * @property int $TaxRateID
 * @method FoxcommerceCurrency Currency()
 * @method FoxcommerceTaxRate TaxRate()
 * @mixin FoxcommerceCountry_TaxExtension
 */
class FoxcommerceCountry extends DataObject
{

    protected static $singular_name = 'Country';
    protected static $plural_name   = 'Countries';

    protected static $db = [
        'Title'      => 'Varchar(255)',
        'Code'       => 'Varchar',
        'LocaleCode' => 'Varchar',
        'Enabled'    => 'Boolean'
    ];

    private static $has_one = [
        'Currency' => 'FoxcommerceCurrency'
    ];

    private static $defaults = [
        'Enabled' => true
    ];

    private static $summary_fields = [
        'Title'         => 'Title',
        'Code'          => 'Code',
        'Currency.Name' => 'Currency',
        'Enabled.Nice'  => 'Enabled'
    ];

    private static $searchable_fields = [
        'Title'      => ['title' => 'Country Title'],
        'Code'       => ['title' => 'Country Code'],
        'CurrencyID' => ['title' => 'Currency'],
        'Enabled'    => ['title' => 'Enabled']
    ];

    private static $cart_summary = [
        'Title',
        'Code',
        'LocaleCode'
    ];

    private static $cart_summary_relations = [
        'Currency'
    ];

    private static $default_sort = 'Title ASC';

    public function getCMSFields()
    {

        $currencies        = FoxcommerceCurrency::get('FoxcommerceCurrency', ['Enabled' => true])->map()->toArray();
        $localeInformation = $this->getLocaleInformation();
        array_walk($localeInformation, function (&$value, $key) {
            $out = "{$value['Language']} - {$value['Country']}";
            if (isset($value['Extra'])) {
                $out .= " ({$value['Extra']})";
            }

            $value = $out;
        });


        $fields = new FieldList([
            TextField::create('Title')->setPlaceholder('New Zealand'),
            TextField::create('Code')->setPlaceholder('NZ'),
            DropdownField::create('LocaleCode', 'Language', $localeInformation)->setDescription(
                _t('FoxcommerceCountry.LocaleListLimit',
                    'If a language is not showing in the list a server administrator will need to install it'
                )
            ),
            DropdownField::create('Enabled', 'Enabled', [1 => 'Enabled', 0 => 'Disabled']),
        ]);

        if ($currencies) {
            $fields->insertBefore('Enabled',
                DropdownField::create('CurrencyID', 'Currency', $currencies)
                    ->setHasEmptyDefault(true)->setEmptyString('(none)')
            );
        }

        if ($this->CurrencyID && !array_key_exists($this->CurrencyID, $currencies)) {
            $obsoleteWarning = _t(
                'FoxcommerceCountry.DisabledCurrencyWarning',
                "This country is linked to a currency that is no longer available. Saving will unlink this country from the currency."
            );
            $fields->insertBefore('Title', LiteralField::create("DisabledCurrencyWarningHeader",
                "<p class=\"message warning\">{$obsoleteWarning}</p>")
            );
        }

        $this->extend('updateCMSFields', $fields);

        return $fields;

    }

    public function getLocaleInformation()
    {

        $localeCodes = explode("\n", shell_exec('locale -a'));

        $countryCodes = [
            'AF' => "Afghanistan",
            'AL' => "Albania",
            'DZ' => "Algeria",
            'AS' => "American Samoa",
            'AD' => "Andorra",
            'AO' => "Angola",
            'AI' => "Anguilla",
            'AQ' => "Antarctica",
            'AG' => "Antigua And Barbuda",
            'AR' => "Argentina",
            'AM' => "Armenia",
            'AW' => "Aruba",
            'AU' => "Australia",
            'AT' => "Austria",
            'AZ' => "Azerbaijan",
            'BS' => "Bahamas",
            'BH' => "Bahrain",
            'BD' => "Bangladesh",
            'BB' => "Barbados",
            'BY' => "Belarus",
            'BE' => "Belgium",
            'BZ' => "Belize",
            'BJ' => "Benin",
            'BM' => "Bermuda",
            'BT' => "Bhutan",
            'BO' => "Bolivia",
            'BA' => "Bosnia And Herzegovina",
            'BW' => "Botswana",
            'BV' => "Bouvet Island",
            'BR' => "Brazil",
            'IO' => "British Indian Ocean Territory",
            'BN' => "Brunei Darussalam",
            'BG' => "Bulgaria",
            'BF' => "Burkina Faso",
            'BI' => "Burundi",
            'KH' => "Cambodia",
            'CM' => "Cameroon",
            'CA' => "Canada",
            'CV' => "Cape Verde",
            'KY' => "Cayman Islands",
            'CF' => "Central African Republic",
            'TD' => "Chad",
            'CL' => "Chile",
            'CN' => "China",
            'CX' => "Christmas Island",
            'CC' => "Cocos (keeling) Islands",
            'CO' => "Colombia",
            'KM' => "Comoros",
            'CG' => "Congo",
            'CD' => "Congo, The Democratic Republic Of The",
            'CK' => "Cook Islands",
            'CR' => "Costa Rica",
            'CI' => "Cote D'ivoire",
            'HR' => "Croatia",
            'CU' => "Cuba",
            'CY' => "Cyprus",
            'CZ' => "Czech Republic",
            'DK' => "Denmark",
            'DJ' => "Djibouti",
            'DM' => "Dominica",
            'DO' => "Dominican Republic",
            'EC' => "Ecuador",
            'EG' => "Egypt",
            'SV' => "El Salvador",
            'GQ' => "Equatorial Guinea",
            'ER' => "Eritrea",
            'EE' => "Estonia",
            'ET' => "Ethiopia",
            'FK' => "Falkland Islands (malvinas)",
            'FO' => "Faroe Islands",
            'FJ' => "Fiji",
            'FI' => "Finland",
            'FR' => "France",
            'GF' => "French Guiana",
            'PF' => "French Polynesia",
            'TF' => "French Southern Territories",
            'GA' => "Gabon",
            'GM' => "Gambia",
            'GE' => "Georgia",
            'DE' => "Germany",
            'GH' => "Ghana",
            'GI' => "Gibraltar",
            'GR' => "Greece",
            'GL' => "Greenland",
            'GD' => "Grenada",
            'GP' => "Guadeloupe",
            'GU' => "Guam",
            'GT' => "Guatemala",
            'GN' => "Guinea",
            'GW' => "Guinea-bissau",
            'GY' => "Guyana",
            'HT' => "Haiti",
            'HM' => "Heard Island And Mcdonald Islands",
            'VA' => "Holy See (vatican City State)",
            'HN' => "Honduras",
            'HK' => "Hong Kong",
            'HU' => "Hungary",
            'IS' => "Iceland",
            'IN' => "India",
            'ID' => "Indonesia",
            'IR' => "Iran, Islamic Republic Of",
            'IQ' => "Iraq",
            'IE' => "Ireland",
            'IL' => "Israel",
            'IT' => "Italy",
            'JM' => "Jamaica",
            'JP' => "Japan",
            'JO' => "Jordan",
            'KZ' => "Kazakhstan",
            'KE' => "Kenya",
            'KI' => "Kiribati",
            'KP' => "Korea, Democratic People's Republic Of",
            'KR' => "Korea, Republic Of",
            'KW' => "Kuwait",
            'KG' => "Kyrgyzstan",
            'LA' => "Lao People's Democratic Republic",
            'LV' => "Latvia",
            'LB' => "Lebanon",
            'LS' => "Lesotho",
            'LR' => "Liberia",
            'LY' => "Libyan Arab Jamahiriya",
            'LI' => "Liechtenstein",
            'LT' => "Lithuania",
            'LU' => "Luxembourg",
            'MO' => "Macao",
            'MK' => "Macedonia, The Former Yugoslav Republic Of",
            'MG' => "Madagascar",
            'MW' => "Malawi",
            'MY' => "Malaysia",
            'MV' => "Maldives",
            'ML' => "Mali",
            'MT' => "Malta",
            'MH' => "Marshall Islands",
            'MQ' => "Martinique",
            'MR' => "Mauritania",
            'MU' => "Mauritius",
            'YT' => "Mayotte",
            'MX' => "Mexico",
            'FM' => "Micronesia, Federated States Of",
            'MD' => "Moldova, Republic Of",
            'MC' => "Monaco",
            'MN' => "Mongolia",
            'MS' => "Montserrat",
            'MA' => "Morocco",
            'MZ' => "Mozambique",
            'MM' => "Myanmar",
            'NA' => "Namibia",
            'NR' => "Nauru",
            'NP' => "Nepal",
            'NL' => "Netherlands",
            'AN' => "Netherlands Antilles",
            'NC' => "New Caledonia",
            'NZ' => "New Zealand",
            'NI' => "Nicaragua",
            'NE' => "Niger",
            'NG' => "Nigeria",
            'NU' => "Niue",
            'NF' => "Norfolk Island",
            'MP' => "Northern Mariana Islands",
            'NO' => "Norway",
            'OM' => "Oman",
            'PK' => "Pakistan",
            'PW' => "Palau",
            'PS' => "Palestinian Territory, Occupied",
            'PA' => "Panama",
            'PG' => "Papua New Guinea",
            'PY' => "Paraguay",
            'PE' => "Peru",
            'PH' => "Philippines",
            'PN' => "Pitcairn",
            'PL' => "Poland",
            'PT' => "Portugal",
            'PR' => "Puerto Rico",
            'QA' => "Qatar",
            'RE' => "Reunion",
            'RO' => "Romania",
            'RU' => "Russian Federation",
            'RW' => "Rwanda",
            'SH' => "Saint Helena",
            'KN' => "Saint Kitts And Nevis",
            'LC' => "Saint Lucia",
            'PM' => "Saint Pierre And Miquelon",
            'VC' => "Saint Vincent And The Grenadines",
            'WS' => "Samoa",
            'SM' => "San Marino",
            'ST' => "Sao Tome And Principe",
            'SA' => "Saudi Arabia",
            'SN' => "Senegal",
            'CS' => "Serbia And Montenegro",
            'SC' => "Seychelles",
            'SL' => "Sierra Leone",
            'SG' => "Singapore",
            'SK' => "Slovakia",
            'SI' => "Slovenia",
            'SB' => "Solomon Islands",
            'SO' => "Somalia",
            'ZA' => "South Africa",
            'GS' => "South Georgia And The South Sandwich Islands",
            'ES' => "Spain",
            'LK' => "Sri Lanka",
            'SD' => "Sudan",
            'SR' => "Suriname",
            'SJ' => "Svalbard And Jan Mayen",
            'SZ' => "Swaziland",
            'SE' => "Sweden",
            'CH' => "Switzerland",
            'SY' => "Syrian Arab Republic",
            'TW' => "Taiwan, Province Of China",
            'TJ' => "Tajikistan",
            'TZ' => "Tanzania, United Republic Of",
            'TH' => "Thailand",
            'TL' => "Timor-leste",
            'TG' => "Togo",
            'TK' => "Tokelau",
            'TO' => "Tonga",
            'TT' => "Trinidad And Tobago",
            'TN' => "Tunisia",
            'TR' => "Turkey",
            'TM' => "Turkmenistan",
            'TC' => "Turks And Caicos Islands",
            'TV' => "Tuvalu",
            'UG' => "Uganda",
            'UA' => "Ukraine",
            'AE' => "United Arab Emirates",
            'GB' => "United Kingdom",
            'US' => "United States",
            'UM' => "United States Minor Outlying Islands",
            'UY' => "Uruguay",
            'UZ' => "Uzbekistan",
            'VU' => "Vanuatu",
            'VE' => "Venezuela",
            'VN' => "Viet Nam",
            'VG' => "Virgin Islands, British",
            'VI' => "Virgin Islands, U.s.",
            'WF' => "Wallis And Futuna",
            'EH' => "Western Sahara",
            'YE' => "Yemen",
            'ZM' => "Zambia",
            'ZW' => "Zimbabwe",
        ];

        $languageCodes = [
            'en' => 'English',
            'aa' => 'Afar',
            'ab' => 'Abkhazian',
            'af' => 'Afrikaans',
            'am' => 'Amharic',
            'ar' => 'Arabic',
            'as' => 'Assamese',
            'ay' => 'Aymara',
            'az' => 'Azerbaijani',
            'ba' => 'Bashkir',
            'be' => 'Byelorussian',
            'bg' => 'Bulgarian',
            'bh' => 'Bihari',
            'bi' => 'Bislama',
            'bn' => 'Bengali/Bangla',
            'bo' => 'Tibetan',
            'br' => 'Breton',
            'ca' => 'Catalan',
            'co' => 'Corsican',
            'cs' => 'Czech',
            'cy' => 'Welsh',
            'da' => 'Danish',
            'de' => 'German',
            'dz' => 'Bhutani',
            'el' => 'Greek',
            'eo' => 'Esperanto',
            'es' => 'Spanish',
            'et' => 'Estonian',
            'eu' => 'Basque',
            'fa' => 'Persian',
            'fi' => 'Finnish',
            'fj' => 'Fiji',
            'fo' => 'Faeroese',
            'fr' => 'French',
            'fy' => 'Frisian',
            'ga' => 'Irish',
            'gd' => 'Scots/Gaelic',
            'gl' => 'Galician',
            'gn' => 'Guarani',
            'gu' => 'Gujarati',
            'ha' => 'Hausa',
            'hi' => 'Hindi',
            'hr' => 'Croatian',
            'hu' => 'Hungarian',
            'hy' => 'Armenian',
            'ia' => 'Interlingua',
            'ie' => 'Interlingue',
            'ik' => 'Inupiak',
            'in' => 'Indonesian',
            'is' => 'Icelandic',
            'it' => 'Italian',
            'iw' => 'Hebrew',
            'ja' => 'Japanese',
            'ji' => 'Yiddish',
            'jw' => 'Javanese',
            'ka' => 'Georgian',
            'kk' => 'Kazakh',
            'kl' => 'Greenlandic',
            'km' => 'Cambodian',
            'kn' => 'Kannada',
            'ko' => 'Korean',
            'ks' => 'Kashmiri',
            'ku' => 'Kurdish',
            'ky' => 'Kirghiz',
            'la' => 'Latin',
            'ln' => 'Lingala',
            'lo' => 'Laothian',
            'lt' => 'Lithuanian',
            'lv' => 'Latvian/Lettish',
            'mg' => 'Malagasy',
            'mi' => 'Maori',
            'mk' => 'Macedonian',
            'ml' => 'Malayalam',
            'mn' => 'Mongolian',
            'mo' => 'Moldavian',
            'mr' => 'Marathi',
            'ms' => 'Malay',
            'mt' => 'Maltese',
            'my' => 'Burmese',
            'na' => 'Nauru',
            'ne' => 'Nepali',
            'nl' => 'Dutch',
            'no' => 'Norwegian',
            'oc' => 'Occitan',
            'om' => '(Afan)/Oromoor/Oriya',
            'pa' => 'Punjabi',
            'pl' => 'Polish',
            'ps' => 'Pashto/Pushto',
            'pt' => 'Portuguese',
            'qu' => 'Quechua',
            'rm' => 'Rhaeto-Romance',
            'rn' => 'Kirundi',
            'ro' => 'Romanian',
            'ru' => 'Russian',
            'rw' => 'Kinyarwanda',
            'sa' => 'Sanskrit',
            'sd' => 'Sindhi',
            'sg' => 'Sangro',
            'sh' => 'Serbo-Croatian',
            'si' => 'Singhalese',
            'sk' => 'Slovak',
            'sl' => 'Slovenian',
            'sm' => 'Samoan',
            'sn' => 'Shona',
            'so' => 'Somali',
            'sq' => 'Albanian',
            'sr' => 'Serbian',
            'ss' => 'Siswati',
            'st' => 'Sesotho',
            'su' => 'Sundanese',
            'sv' => 'Swedish',
            'sw' => 'Swahili',
            'ta' => 'Tamil',
            'te' => 'Tegulu',
            'tg' => 'Tajik',
            'th' => 'Thai',
            'ti' => 'Tigrinya',
            'tk' => 'Turkmen',
            'tl' => 'Tagalog',
            'tn' => 'Setswana',
            'to' => 'Tonga',
            'tr' => 'Turkish',
            'ts' => 'Tsonga',
            'tt' => 'Tatar',
            'tw' => 'Twi',
            'uk' => 'Ukrainian',
            'ur' => 'Urdu',
            'uz' => 'Uzbek',
            'vi' => 'Vietnamese',
            'vo' => 'Volapuk',
            'wo' => 'Wolof',
            'xh' => 'Xhosa',
            'yo' => 'Yoruba',
            'zh' => 'Chinese',
            'zu' => 'Zulu',
        ];

        $localeData = [];

        foreach ($localeCodes as $c => $l) {
            if (strlen($l)) {
                $parts = explode('.', $l);
                $lc    = $parts[0];

                if ($lc === 'C' || $lc === 'POSIX') {
                    continue;
                }

                list($lcode, $ccode) = explode('_', $lc);

                $lcode = strtolower($lcode);

                $language = $languageCodes[$lcode];
                $country  = $countryCodes[$ccode];

                if (strlen($language) and strlen($country)) {
                    $localeData[$l] = array_filter([
                        "Language" => $language,
                        "Country"  => $country,
                        "Extra"    => isset($parts[1]) ? $parts[1] : null
                    ]);
                }

            }
        }
        return $localeData;
    }

    public function canCreate($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canView($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canEdit($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canDelete($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

}