<?php

/**
 * Class FoxcommerceAddress
 *
 * @property string $AddressLine1
 * @property string $AddressLine2
 * @property string $Suburb
 * @property string $Region
 * @property string $Postcode
 * @property string $Country
 * @property string $Usage
 * @property boolean $DefaultForBilling
 * @property boolean $DefaultForShipping
 * @property boolean $Archived
 * @property int $MemberID
 * @method Member Member()
 */
class FoxcommerceAddress extends DataObject
{
    protected static $singular_name = 'Address';
    protected static $plural_name   = 'Addresses';

    protected static $db = [
        'AddressLine1'       => 'Varchar(255)',
        'AddressLine2'       => 'Varchar(255)',
        'Suburb'             => 'Varchar(255)',
        'Region'             => 'Varchar(255)',
        'Postcode'           => 'Varchar(255)',
        'Country'            => 'Varchar(255)',
        'Usage'              => 'Enum("Billing, Shipping, Both", "Both")',
        'DefaultForBilling'  => 'Boolean',
        'DefaultForShipping' => 'Boolean',
        'Archived'           => 'Boolean'
    ];

    protected static $has_one = [
        'Member' => 'Member'
    ];

    protected static $summary_fields = [
        'AddressLine1' => 'Address line one',
        'AddressLine2' => 'Address line two',
        'Suburb'       => 'Suburb',
        'Region'       => 'Region',
        'Postcode'     => 'Postcode',
        'Country'      => 'Country',
        'Usage'        => 'Usage'
    ];

    public function getTitle()
    {
        return implode(', ', array_filter([
            $this->AddressLine1,
            $this->AddressLine2,
            $this->Suburb,
            $this->Region,
            $this->Postcode,
            $this->Country
        ]));
    }

    public function usageOptions()
    {
        $usageOptions = $this->dbObject('Usage')->EnumValues();

        if ($this->DefaultForShipping && $this->DefaultForBilling) {
            $usageOptions = ['Both' => 'Both'];
        } elseif ($this->DefaultForShipping) {
            $usageOptions = ['Shipping' => 'Shipping', 'Both' => 'Both'];
        } elseif ($this->DefaultForBilling) {
            $usageOptions = ['Billing' => 'Billing', 'Both' => 'Both'];
        }

        return $usageOptions;
    }

    public function getCMSFields()
    {
        $this->beforeUpdateCMSFields(function (FieldList $fields) {
            $fields->removeByName('MemberID');

            $fields->replaceField('Country', CountryDropdownField::create('Country'));
            $fields->replaceField('Usage', DropdownField::create('Usage', 'Usage', $this->usageOptions()));
            $fields->replaceField('Archived', DropdownField::create('Archived', 'Status', [0 => 'Available', 1 => 'Archived']));
        });

        return parent::getCMSFields();
    }

    public function getFrontEndFields($params = null)
    {

        $fields = FieldList::create([
            HiddenField::create('ID'),
            TextField::create('AddressLine1', 'Address Line 1'),
            TextField::create('AddressLine2', 'Address Line 2'),
            TextField::create('Suburb'),
            TextField::create('Region'),
            TextField::create('Postcode'),
            CountryDropdownField::create('Country'),
            DropdownField::create('Usage')->setSource($this->usageOptions()),
        ]);


        $this->extend('updateFrontEndFields', $fields);
        return $fields;
    }


    public function IsForShipping()
    {
        return in_array($this->Usage, ['Shipping', 'Both']);
    }

    public function IsForBilling()
    {
        return in_array($this->Usage, ['Billing', 'Both']);
    }

    public function onBeforeWrite()
    {
        parent::onBeforeWrite();

        if ($this->Archived) {
            $this->DefaultForBilling  = false;
            $this->DefaultForShipping = false;
        } else {
            if ($this->isChanged('DefaultForBilling') && $this->DefaultForBilling) {
                $this->unDefaultOthers('Billing');
            }

            if ($this->isChanged('DefaultForShipping') && $this->DefaultForShipping) {
                $this->unDefaultOthers('Shipping');
            }
        }

    }

    public function CanBeArchived()
    {
        if ($this->DefaultForShipping || $this->DefaultForBilling) return false;
        if ($this->IsForShipping() && $this->Member()->ShippingAddresses()->Count() === 1) return false;
        if ($this->IsForBilling() && $this->Member()->BillingAddresses()->Count() === 1) return false;
        return true;
    }

    public function unDefaultOthers($type)
    {
        if (!$this->ID || !$this->MemberID) return;

        /** @var HasManyList $addresses */
        $addresses = $this->Member()->AllAddresses(true)->filter(["DefaultFor{$type}" => true])->exclude(['ID' => $this->ID]);

        if ($addresses) {
            $addresses->each(function (FoxcommerceAddress $address) use ($type) {
                $address->{"DefaultFor{$type}"} = false;
                $address->write();
            });
        }
    }

    public function canView($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canEdit($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canDelete($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canCreate($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

}