<?php

class FoxcommerceSaveDetailsField extends FormField
{
    /**
     * Minimum character length of the password.
     *
     * @var int
     */
    public $minLength = null;

    /**
     * Maximum character length of the password.
     *
     * @var int
     */
    public $maxLength = null;

    /**
     * Enforces at least one digit and one alphanumeric
     * character (in addition to {$minLength} and {$maxLength}
     *
     * @var boolean
     */
    public $requireStrongPassword = false;

    /**
     * A place to temporarily store the confirm password value
     *
     * @var string
     */
    protected $confirmValue;

    /**
     * A place to temporarily store the is visible value
     *
     * @var string
     */
    protected $isVisibleValue;

    /**
     * Child fields (_Password, _ConfirmPassword)
     *
     * @var FieldList
     */
    public $children;

    /**
     * @param string $name
     * @param string $title
     * @param mixed  $value
     * @param Form   $form
     * @param string $titleShowOnClick
     * @param string $titleConfirmField Alternate title (not localizeable)
     */
    public function __construct($name, $title = null, $value = "", $form = null, $titleShowOnClick = null, $titleConfirmField = null)
    {

        // naming with underscores to prevent values from actually being saved somewhere
        $this->children = new FieldList(
            new CheckboxField(
                "{$name}[_PasswordFieldVisible]",
                $titleShowOnClick ?: _t('Member.WANTACCOUNT', 'Save my details for later')
            ),
            new PasswordField(
                "{$name}[_Password]",
                (isset($title)) ? $title : _t('Member.PASSWORD', 'Password')
            ),
            new PasswordField(
                "{$name}[_ConfirmPassword]",
                (isset($titleConfirmField)) ? $titleConfirmField : _t('Member.CONFIRMPASSWORD', 'Confirm Password')
            )
        );

        // disable auto complete
        foreach ($this->children as $child) {
            /** @var FormField $child */
            $child->setAttribute('autocomplete', 'off');
        }

        // we have labels for the subfields
        $title = false;

        parent::__construct($name, $title);
        $this->setValue($value);
    }

    /**
     * @param array $properties
     *
     * @return HTMLText|string
     */
    public function Field($properties = array())
    {
        Requirements::javascript(FOXCOMMERCE_DIR . '/javascript/SaveDetailsField.js');

        $content = '';

        /** @var CheckboxField $toggleField */
        $toggleField = $this->children->first();
        $toggleField->setDisabled($this->isDisabled());
        $toggleField->setReadonly($this->isReadonly());
        $toggleField->setAttribute('onclick', 'saveDetailsFieldToggle(this)');

        $isVisible = $this->children->fieldByName($this->getName() . '[_PasswordFieldVisible]');
        if ($this->message) $isVisible->setValue(1);

        $displayStyle = (($isVisible && $isVisible->Value())) ? 'block' : 'none';
        $content .= $toggleField->FieldHolder();

        $content .= '<div class="saveDetailsFieldContainer" style="display:' . $displayStyle . ';">';

        foreach ($this->children as $index => $field) {
            if ($index === 0) continue;

            /** @var FormField $field */
            $field->setDisabled($this->isDisabled());
            $field->setReadonly($this->isReadonly());
            $field->setValue('');

            if (count($this->attributes)) {
                foreach ($this->attributes as $name => $value) {
                    $field->setAttribute($name, $value);
                }
            }

            $content .= $field->FieldHolder();
        }

        $content .= "</div>\n";

        return $content;
    }

    /**
     * Returns the children of this field for use in templating.
     * @return FieldList
     */
    public function getChildren()
    {
        return $this->children;
    }

    /**
     * @param string $title
     *
     * @return FoxcommerceSaveDetailsField
     */
    public function setRightTitle($title)
    {
        foreach ($this->children as $field) {
            /** @var FormField $field */
            $field->setRightTitle($title);
        }

        return $this;
    }

    /**
     * Set child field titles. Titles in order should be:
     *  - "Current Password" (if getRequireExistingPassword() is set)
     *  - "Password"
     *  - "Confirm Password"
     *
     * @param array $titles List of child titles
     * @return $this
     */
    public function setChildrenTitles($titles)
    {
        $expectedChildren = 3;
        if (is_array($titles) && count($titles) == $expectedChildren) {
            foreach ($this->children as $field) {
                if (isset($titles[0])) {
                    /** @var FormField $field */
                    $field->setTitle($titles[0]);

                    array_shift($titles);
                }
            }
        }

        return $this;
    }

    /**
     * Value is sometimes an array, and sometimes a single value, so we need
     * to handle both cases.
     *
     * @param mixed $value
     * @param mixed $data
     * @return $this
     */
    public function setValue($value, $data = null)
    {

        // If $data is a DataObject, don't use the value, since it's a hashed value
        if ($data && $data instanceof DataObject) $value = '';

        //store this for later
        $oldValue = $this->value;

        if (is_array($value)) {
            $this->value          = $value['_Password'];
            $this->confirmValue   = $value['_ConfirmPassword'];
            $this->isVisibleValue = isset($value['_PasswordFieldVisible']) && $value['_PasswordFieldVisible'];

            if (isset($value['_PasswordFieldVisible'])) {
                $this->children->fieldByName($this->getName() . '[_PasswordFieldVisible]')
                    ->setValue($this->isVisibleValue);
            }
        } else {
            if ($value) {
                $this->value          = $value;
                $this->confirmValue   = $value;
                $this->isVisibleValue = true;
            }
        }

        if ($oldValue != $this->value) {
            $this->children->fieldByName($this->getName() . '[_Password]')
                ->setValue($this->value);

            $this->children->fieldByName($this->getName() . '[_ConfirmPassword]')
                ->setValue($this->confirmValue);
        }

        return $this;
    }

    /**
     * Update the names of the child fields when updating name of field.
     *
     * @param string $name new name to give to the field.
     * @return $this
     */
    public function setName($name)
    {
        $this->children->fieldByName($this->getName() . '[_Password]')
            ->setName($name . '[_Password]');
        $this->children->fieldByName($this->getName() . '[_ConfirmPassword]')
            ->setName($name . '[_ConfirmPassword]');

        return parent::setName($name);
    }

    /**
     * Determines if the field was actually shown on the client side - if not,
     * we don't validate or save it.
     *
     * @return boolean
     */
    public function isSaveable()
    {
        $isVisible = $this->children->fieldByName($this->getName() . '[_PasswordFieldVisible]');

        return ($isVisible && $isVisible->Value());
    }

    /**
     * Validate this field
     *
     * @param Validator $validator
     * @return bool
     */
    public function validate($validator)
    {
        $name = $this->name;

        // if field isn't visible, don't validate
        if (!$this->isSaveable()) {
            return true;
        }

        $passwordField        = $this->children->fieldByName($name . '[_Password]');
        $passwordConfirmField = $this->children->fieldByName($name . '[_ConfirmPassword]');
        $passwordField->setValue($this->value);
        $passwordConfirmField->setValue($this->confirmValue);

        $value = $passwordField->Value();

        // both password-fields should be the same
        if ($value != $passwordConfirmField->Value()) {
            $validator->validationError(
                $name,
                _t('Form.VALIDATIONPASSWORDSDONTMATCH', "Passwords don't match"),
                "validation"
            );

            return false;
        }

        // both password-fields shouldn't be empty
        if (!$value || !$passwordConfirmField->Value()) {
            $validator->validationError(
                $name,
                _t('Form.VALIDATIONPASSWORDSNOTEMPTY', "Passwords can't be empty"),
                "validation"
            );

            return false;
        }

        // lengths
        if (($this->minLength || $this->maxLength)) {
            $errorMsg = null;
            $limit    = null;
            if ($this->minLength && $this->maxLength) {
                $limit    = "{{$this->minLength},{$this->maxLength}}";
                $errorMsg = _t(
                    'ConfirmedPasswordField.BETWEEN',
                    'Passwords must be {min} to {max} characters long.',
                    array('min' => $this->minLength, 'max' => $this->maxLength)
                );
            } elseif ($this->minLength) {
                $limit    = "{{$this->minLength}}.*";
                $errorMsg = _t(
                    'ConfirmedPasswordField.ATLEAST',
                    'Passwords must be at least {min} characters long.',
                    array('min' => $this->minLength)
                );
            } elseif ($this->maxLength) {
                $limit    = "{0,{$this->maxLength}}";
                $errorMsg = _t(
                    'ConfirmedPasswordField.MAXIMUM',
                    'Passwords must be at most {max} characters long.',
                    array('max' => $this->maxLength)
                );
            }
            $limitRegex = '/^.' . $limit . '$/';
            if (!empty($value) && !preg_match($limitRegex, $value)) {
                $validator->validationError(
                    $name,
                    $errorMsg,
                    "validation"
                );
            }
        }

        if ($this->requireStrongPassword) {
            if (!preg_match('/^(([a-zA-Z]+\d+)|(\d+[a-zA-Z]+))[a-zA-Z0-9]*$/', $value)) {
                $validator->validationError(
                    $name,
                    _t('Form.VALIDATIONSTRONGPASSWORD',
                        "Passwords must have at least one digit and one alphanumeric character"),
                    "validation"
                );

                return false;
            }
        }

        return true;
    }

    /**
     * Only save if field was shown on the client, and is not empty.
     *
     * @param DataObjectInterface $record
     *
     * @return boolean
     */
    public function saveInto(DataObjectInterface $record)
    {
        if (!$this->isSaveable()) {
            return false;
        }

        if ($this->value) {
            parent::saveInto($record);
        }
    }

    /**
     * Makes a read only field with some stars in it to replace the password
     *
     * @return ReadonlyField
     */
    public function performReadonlyTransformation()
    {
        $field = $this->castedCopy('ReadonlyField')
            ->setTitle($this->title ? $this->title : _t('Member.PASSWORD'))
            ->setValue('*****');

        return $field;
    }

}