<?php

class FoxcommerceAddressForm extends Form
{

    protected static $allowed_actions = [
        'submit',
        'addnew',
        'defaultshipping',
        'defaultbilling'
    ];

    protected $override_usage = false;

    public function __construct(Controller $controller, $name)
    {
        $validator = new RequiredFields($this->getRequiredFields());

        $emptyAddress = FoxcommerceAddress::create();

        //We do things a bit odd here, normally you'd pass through the form fields and actions
        //but in our case we need to know about the address to construct them properly
        //So we hold off on setting the real fields until we populate the form with data
        parent::__construct($controller, $name, $this->getFields($emptyAddress), $this->getActions($emptyAddress), $validator);
    }

    public function getRequiredFields()
    {

        $fields = [];

        //Make sure the listener uses pass by reference I.e public function updateRequiredFields(&$fields) { ... }
        $this->extend('updateRequiredFields', $fields);

        return $fields;
    }

    protected function getFields(FoxcommerceAddress $address)
    {

        /** @var FieldList $fields */
        $fields = $address->getFrontEndFields();

        $existingAddresses = Member::currentUser()->AllAddresses();
        $billingAddresses  = Member::currentUser()->BillingAddresses();
        $shippingAddresses = Member::currentUser()->ShippingAddresses();

        if (!$address->exists()) {
            if ($existingAddresses->count() === 0) {
                $fields->replaceField('Usage', HiddenField::create('Usage', 'Usage')->setValue($this->override_usage = 'Both'));
                $fields->insertAfter('Usage', new LiteralField('UsageHardSet',
                    '<p class="message info">As this is your first address we will set this as your default billing and shipping address.</p>'
                ));
            } elseif ($billingAddresses->count() === 0) {
                /** @var DropdownField $usageField */
                $usageField = $fields->fieldByName('Usage');
                $usageField->setSource(['Billing' => 'Billing', 'Both' => 'Both']);
                $usageField->setValue($this->override_usage = 'Billing');
                $fields->insertAfter('Usage', new LiteralField('UsageHardSet',
                    '<p class="message info">As you do not have a billing address setup we will set this as your default billing address.</p>'
                ));
            } elseif ($shippingAddresses->count() === 0) {
                /** @var DropdownField $usageField */
                $usageField = $fields->fieldByName('Usage');
                $usageField->setSource(['Shipping' => 'Shipping', 'Both' => 'Both']);
                $usageField->setValue($this->override_usage = 'Shipping');
                $fields->insertAfter('Usage', new LiteralField('UsageHardSet',
                    '<p class="message info">As you do not have a shipping address setup we will set this as your default shipping address.</p>'
                ));
            }
        }

        $this->extend('updateFields', $fields);

        return $fields;
    }

    protected function getActions(FoxcommerceAddress $address)
    {

        if ($address->exists()) {

            $actions = FieldList::create([
                FormAction::create('submit', 'Save Changes')
            ]);

            if ($address->IsForShipping() && !$address->DefaultForShipping) {
                $actions->push(FormAction::create('defaultshipping', 'Set as default shipping address'));
            }

            if ($address->IsForBilling() && !$address->DefaultForBilling) {
                $actions->push(FormAction::create('defaultbilling', 'Set as default billing address'));
            }
        } else {
            $actions = FieldList::create([
                FormAction::create('submit', 'Save')
            ]);
        }


        $this->extend('updateActions', $actions);

        return $actions;
    }

    public function submit($data, FoxcommerceAddressForm $form, SS_HTTPRequest $request)
    {

        $defaultBilling    = Member::currentUser()->DefaultBillingAddress();
        $defaultShipping   = Member::currentUser()->DefaultShippingAddress();

        if (isset($data['ID']) && !!$data['ID']) {
            /** @var FoxcommerceAddress $address */
            $address = Member::currentUser()->AllAddresses()->byID($data['ID']);

            if (!$address) {
                $this->controller->httpError(404);
            }

            $form->saveInto($address);

            if (!$defaultBilling && $address->IsForBilling()) {
                $address->DefaultForBilling = true;
            }

            if (!$defaultShipping && $address->IsForShipping()) {
                $address->DefaultForShipping = true;
            }

            $address->write();

            $this->sessionMessage('Your address details have been updated successfully', 'good');
            return $this->controller->redirect(FoxcommerceAccountPage::getInstance()->Link('my-addresses'));
        } else {
            $address = new FoxcommerceAddress();
            $form->saveInto($address);
            $address->MemberID = Member::currentUserID();

            if (!$defaultBilling && $address->IsForBilling()) {
                $address->DefaultForBilling = true;
            }

            if (!$defaultShipping && $address->IsForShipping()) {
                $address->DefaultForShipping = true;
            }

            $recordID = $address->write();

            $this->sessionMessage('Your address was saved successfully', 'good');
            return $this->controller->redirect(FoxcommerceAccountPage::getInstance()->Link('my-addresses'));
        }

    }

    public function defaultshipping($data, FoxcommerceAddressForm $form, SS_HTTPRequest $request)
    {

        if (isset($data['ID'])) {
            $address = Member::currentUser()->ShippingAddresses()->byID($data['ID']);

            if (!$address) {
                $this->controller->httpError(404);
            }

            $address->DefaultForShipping = true;
            $address->write();

            $this->sessionMessage('Default shipping address set successfully', 'good');

            return $this->controller->redirect(FoxcommerceAccountPage::getInstance()->Link('my-addresses'));
        }

        $this->controller->httpError(404);

    }

    public function defaultbilling($data, FoxcommerceAddressForm $form, SS_HTTPRequest $request)
    {


        if (isset($data['ID'])) {
            $address = Member::currentUser()->BillingAddresses()->byID($data['ID']);

            if (!$address) {
                $this->controller->httpError(404);
            }

            $address->DefaultForBilling = true;
            $address->write();

            $this->sessionMessage('Default billing address set successfully', 'good');

            return $this->controller->redirect(FoxcommerceAccountPage::getInstance()->Link('my-addresses'));
        }

        $this->controller->httpError(404);

    }

    public function loadDataFrom($data, $mergeStrategy = 0, $fieldList = null)
    {
        if (is_array($data) && isset($data['ID']) && !!$data['ID']) {
            $address = Member::currentUser()->AllAddresses()->byID($data['ID']);

        } elseif (is_object($data) && $data->ID) {
            $address = Member::currentUser()->AllAddresses()->byID($data->ID);

        } else {
            $address = new FoxcommerceAddress();
        }

        if (!$address) {
            $this->controller->httpError(404);
        }

        $this->setFields($this->getFields($address));
        $this->setActions($this->getActions($address));

        parent::loadDataFrom($data, $mergeStrategy, $fieldList);

        if($this->override_usage){
            $this->Fields()->fieldByName('Usage')->setValue($this->override_usage);
        }

        return $this;


    }


}