<?php


/**
 * Class FoxcommerceCartController
 *
 */
class FoxcommerceCartController extends Controller
{

  private static $bad_bots = [
    'AhrefsBot',
    'Baiduspider',
    'Googlebot',
    'MJ12bot',
    'DotBot',
    'bingbot',
    'YandexBot',
  ];

  private static $allowed_actions = [
    'update',
    'add',
    'remove',
    'items',
    'reassociate',
    'setcountry',
    'clear',
    'cart',
    'showproduct',
    'session',
    'claim',
  ];

  public function init()
  {
    if (isset($_SERVER['HTTP_USER_AGENT'])) {
      foreach (self::config()->get('bad_bots') as $bot) {
        if (strpos($_SERVER['HTTP_USER_AGENT'], $bot) !== false) {
          return $this->httpError(403, 'User agent appears to be a bot');
        }
      }
    }
    parent::init();
  }

  /******************************************************
   * Cart Display Methods
   ******************************************************/

  public function session(SS_HTTPRequest $request)
  {
    return $this->jsonSuccessNew(['Value' => Session::get('order-guid'), 'Action' => 'session']);
  }

  public function claim(SS_HTTPRequest $request)
  {
//        if (!Director::isDev() && !Permission::check('ADMIN')) $this->httpError(403);

    if (!$request->param('ID')) {
      return $this->jsonError(['Message' => 'Missing order session ID']);
    }

    $order = FoxcommerceOrder::get()->filter(['Closed' => false, 'SessionID' => $request->param('ID')])->first();
    if (!$order) return $this->jsonError(['Message' => 'Invalid order session ID']);
    Session::set('order-guid', FoxcommerceHelper::guid());
    $order->SessionID = Session::get('order-guid');
    $order->write();

    FoxcommerceOrder::setCurrent($order);

    return $this->jsonSuccess(['Message' => 'Cart claimed successfully', 'Action' => 'claim']);
  }

  public function cart(SS_HTTPRequest $request)
  {

    $request = $this->transformRequest($request);

    $order = FoxcommerceOrder::current();
    if ($order->exists() && $request->requestVar('refresh') !== null) {
      //force recalculate totals
      $order->write(false, false, true);
      return $this->jsonSuccess(['Message' => 'Cart Refreshed', 'Action' => 'display']);
    }

    return $this->jsonSuccess(['Message' => '', 'Action' => 'display']);
  }

  public function showproduct(SS_HTTPRequest $request, $returnRaw = false)
  {
    $request = $this->transformRequest($request);

    if (!$request->requestVar('ProductID')) {
      return $this->jsonError(['Message' => 'Missing "ProductID" requestVar']);
    }

    /** @var FoxcommerceProduct $product */
    $product = FoxcommerceProduct::get_by_id('FoxcommerceProduct', $request->requestVar('ProductID'));

    if (!$product) {
      return $this->jsonError(['Message' => 'Invalid "ProductID" requestVar']);
    }

    $productSummary                   = FoxcommerceHelper::build_cart_summary($product);
    $productSummary['QuantityInCart'] = 0;

    $order = FoxcommerceOrder::current();

    if ($order->exists()) {
      $filterData              = $request->requestVar('CartFilterData') ?: [];
      $filterData['ProductID'] = $product->ID;
      if ($order->hasItem($filterData)) {
        $productSummary['QuantityInCart'] = $order->getItem($filterData)->Quantity;
      }
    }

    if ($returnRaw) {
      return $productSummary;
    }

    return $this->jsonSuccessNew($productSummary);
  }

  /******************************************************
   * Order Manipulation
   ******************************************************/

  public function clear()
  {
    FoxcommerceOrder::current()->close();
    return $this->jsonSuccess(['Message' => 'Cart order closed', 'Action' => 'clear']);
  }

  public function setcountry(SS_HTTPRequest $request)
  {
    $request = $this->transformRequest($request);

    if (!$request->param('ID')) {
      return $this->jsonError(['Message' => 'Invalid or missing "CountryCode"']);
    }

    /** @var FoxcommerceCountry $country */
    $country = FoxcommerceCountry::get_one('FoxcommerceCountry', ['Code' => $request->param('ID')]);
    if (!$country) {
      return $this->jsonError(['Message' => 'Non matching "CountryCode" parameter']);
    }

    $currentOrder = FoxcommerceOrder::current();

    $currentOrder->CountryID = $country->ID;
    $currentOrder->write();

    return $this->jsonSuccess(['Message' => 'Cart country set to ' . $country->Title, 'Action' => 'updatecountry']);
  }

  /******************************************************
   * Cart Manipulation
   ******************************************************/

  /**
   * Updates the quantity of the product in the cart
   * If the product is not in the cart it will add it
   *
   * @param \SS_HTTPRequest $request
   * @return \SS_HTTPResponse
   */
  public function update(SS_HTTPRequest $request)
  {
    // Don't allow modification of already paid orders
    $order   = FoxcommerceOrder::current();
    if($order->getIsPaid()) {
      $order->disassociate();
    }
    $request = $this->transformRequest($request);

    $productID = $request->requestVar('ProductID');
    $quantity  = $request->requestVar('Quantity');

    if (!$productID || !ctype_digit((string)$productID)) {
      return $this->jsonError(['Message' => 'Invalid or missing "ProductID" parameter'], 400);
    }

    if ($quantity === false || !ctype_digit((string)$quantity)) {
      return $this->jsonError(['Message' => 'Invalid or missing "Quantity" parameter'], 400);
    }

    /** @var FoxcommerceProduct $product */
    if (!$product = FoxcommerceProduct::get_by_id('FoxcommerceProduct', $productID)) {
      return $this->jsonError(['message' => 'Non matching "ProductID" parameter'], 400);
    }

    $beforeUpdateCheck = $order->extend('beforeUpdateOrderRequest', $request, $product);
    if (!empty($beforeUpdateCheck)) {
      //there may be multiple reasons these were blocked, but we'll just return the first one
      if ($beforeUpdateCheck[0] instanceof SS_HTTPResponse) {
        return $beforeUpdateCheck[0];
      } elseif (is_array($beforeUpdateCheck[0]) && isset($beforeUpdateCheck[0]['code'])) {
        return $this->jsonError(['message' => $beforeUpdateCheck[0]['message']], $beforeUpdateCheck[0]['code']);
      } else {
        return $this->jsonError(['message' => $beforeUpdateCheck[0]], 400);
      }
    }

    if (!$order->isInDB()) {
      $order->write();
    }

    $filterData = FoxcommerceHelper::array_except($request->requestVars(), ['url', 'Quantity']);
    $order->extend('updateFilterData', $filterData);
    $requestedQuantity = intval($quantity);
    $orderItem         = $order->getItem($filterData);

    if ($requestedQuantity === 0 && !$orderItem) {
      $message = 'Product removed from cart';
      $action  = 'remove';
    } elseif ($requestedQuantity === 0) {
      $message = 'Product removed from cart';
      $action  = 'remove';
    } elseif ($orderItem) {
      $message = 'Product quantity updated';
      $action  = 'update';
    } else {
      $message = 'Product added to cart';
      $action  = 'add';
      /** @var FoxcommerceOrderItem $orderItem */
      $orderItem                   = FoxcommerceOrderItem::create($filterData);
      $orderItem->OrderID          = $order->ID;
      $orderItem->ProductVersionID = $product->Version;

      $order->extend('beforeWriteOrderItem', $request, $orderItem);

      $orderItem->write();

      $order->extend('afterWriteOrderItem', $request, $orderItem);

      /** @var FoxcommerceProductAttributeValueLink $attributeValueLink */
      foreach ($product->AttributeValues() as $attributeValueLink) {
        /** @var FoxcommerceOrderItemAttributeValueLink $orderItemAttributeValue */

        $attribute = $attributeValueLink->Attribute();
        $value     = $attributeValueLink->AttributeValue();

        $orderItemAttributeValue                          = FoxcommerceOrderItemAttributeValueLink::create();
        $orderItemAttributeValue->OrderItemID             = $orderItem->ID;
        $orderItemAttributeValue->AttributeID             = $attribute->ID;
        $orderItemAttributeValue->AttributeVersionID      = $attribute->Version;
        $orderItemAttributeValue->AttributeValueID        = $value->ID;
        $orderItemAttributeValue->AttributeValueVersionID = $value->Version;
        $orderItemAttributeValue->write();
      }
    }

    if (isset($orderItem)) {
      $orderItem->updateQuantity($quantity);
    }

    //force recalculate totals
    $order->write(false, false, true);

    $afterUpdateCheck = $order->extend('afterUpdateOrderRequest', $request, $product, $action, $message);
    if (!empty($afterUpdateCheck) && $afterUpdateCheck[0] instanceof SS_HTTPResponse) {
      return $afterUpdateCheck[0];
    }

    return $this->jsonSuccess(['Message' => $message, 'Action' => $action]);
  }

  /**
   * Alias method to update
   *
   * @param \SS_HTTPRequest $request
   * @return \SS_HTTPResponse
   */
  public function add(SS_HTTPRequest $request)
  {
    return $this->update($request);
  }

  public function remove(SS_HTTPRequest $request)
  {
    $request = $this->transformRequest($request);

    $postVars             = $request->postVars();
    $postVars['Quantity'] = 0;

    //mock the request again so we can call it with a quantity of zero
    $request = new SS_HTTPRequest($request->httpMethod(), $request->getURL(), $request->getVars(), $postVars,
      $request->getBody());
    return $this->update($request);
  }


  /******************************************************
   * Output Helpers
   ******************************************************/

  public function jsonSuccessNew($data)
  {
    return $this->jsonResponse(['Success' => true, 'Data' => $data], false);
  }

  public function jsonSuccess($metaData)
  {
    return $this->jsonResponse(['Success' => true, 'MetaData' => $metaData]);
  }

  public function jsonError($metaData, $responseCode = 200)
  {
    return $this->jsonResponse(['Success' => false, 'MetaData' => $metaData], false, $responseCode);
  }

  public function jsonResponse(array $content, $addCart = true, $responseCode = 200)
  {
    $this->response->addHeader('Content-Type', 'application/json');
    $this->response->setStatusCode($responseCode);

    $this->timeStart = microtime(true);

    if ($addCart) {
      $content['Cart'] = FoxcommerceHelper::initial_state();
    }

    $this->response->setBody(json_encode($content));
    return $this->response;
  }

  /**
   * Transforms a JSON request to a standard get/post request if required
   *
   * @param SS_HTTPRequest $request
   * @return SS_HTTPRequest
   */
  public function transformRequest(SS_HTTPRequest $request)
  {

    if (FoxcommerceHelper::str_starts_with($request->getHeader('Content-Type'), 'application/json') && $request->getBody()) {
      $getVars  = $request->getVars();
      $postVars = $request->postVars();

      if (in_array($request->httpMethod(), ['GET', 'PATCH', 'DELETE'])) {
        $getVars = array_merge($getVars, json_decode($request->getBody(), true));
      } elseif (in_array($request->httpMethod(), ['POST', 'PUT'])) {
        $postVars = array_merge($postVars, json_decode($request->getBody(), true));
      }

      $request = new SS_HTTPRequest($request->httpMethod(), $request->getURL(), $getVars, $postVars, $request->getBody());
    }

    return $request;
  }
}