<?php

class CheckoutStepCustomerDetails extends CheckoutStep
{

    public function FormFields(Controller $controller = null)
    {
        /** @var Member $member */
        $member = Member::currentUser() ?: Member::create();
        /** @var FoxcommerceAccountPage $accountPage */
        $accountPage = FoxcommerceAccountPage::get_one('FoxcommerceAccountPage');

        if ($member->exists()) {
            $fields = [
                ReadonlyField::create('CustomerFirstName', 'First Name', $member->FirstName),
                ReadonlyField::create('CustomerSurname', 'Surname', $member->Surname),
                ReadonlyField::create('CustomerEmail', 'Email Address', $member->Email),
                LiteralField::create('DetailsLoadedFromDB', '
                    <div class="message info">
                        <strong>We\'ve loaded your existing details from your account.</strong>
                        <p><a href="'.$accountPage->Link().'" class="update-details">Update your details on your account</a></p>
                    </div>
                ')
            ];
        } else {
            $fields = [
                TextField::create('CustomerFirstName', 'First Name', $this->currentOrder->CustomerFirstName),
                TextField::create('CustomerSurname', 'Surname', $this->currentOrder->CustomerSurname),
                EmailField::create('CustomerEmail', 'Email Address', $this->currentOrder->CustomerEmail),
                FoxcommerceSaveDetailsField::create('CustomerPassword')
            ];
        }

        return new FieldList($fields);
    }

    public function Validator()
    {
        return new CheckoutStepCustomerDetailsValidator(Member::currentUser() ? [] : ['CustomerFirstName', 'CustomerSurname', 'CustomerEmail']);
    }

    public function handleSubmission(array $data, CheckoutForm $form, SS_HTTPRequest $request = null)
    {
        $order = $this->currentOrder;
        if (!Member::currentUser()) {
            $order->CustomerFirstName = $data['CustomerFirstName'];
            $order->CustomerSurname   = $data['CustomerSurname'];
            $order->CustomerEmail     = $data['CustomerEmail'];

            if (FoxcommerceHelper::array_get($data, 'CustomerPassword._PasswordFieldVisible', false)) {
                $hashingAlgorithm                 = Security::config()->password_encryption_algorithm;
                $hash                             = FoxcommerceHelper::array_only(
                    Security::encrypt_password($data['CustomerPassword']['_Password'], null, $hashingAlgorithm),
                    ['password', 'salt', 'algorithm']
                );
                $order->CustomerTemporaryPassword = serialize($hash);
            } else {
                $order->CustomerTemporaryPassword = null;
            }
        } else {
            $member                   = Member::currentUser();
            $order->CustomerFirstName = $member->FirstName;
            $order->CustomerSurname   = $member->Surname;
            $order->CustomerEmail     = $member->Email;
            $order->MemberID          = $member->ID;
        }
        $order->write();
        $form->sessionMessage("Welcome {$order->CustomerFirstName}, Just a few more steps!", 'good');
        return true;
    }

    public function hasBeenPassed()
    {
        $order = $this->currentOrder;
        return $order->CustomerFirstName && $order->CustomerEmail;
    }

    public static function extraStatics()
    {
        return [
            'db'             => [
                'CustomerTemporaryPassword' => 'Text'
            ],
            'has_one'        => [
                'Member' => 'Member'
            ],
            'payment_fields' => [
                'firstName' => 'CustomerFirstName',
                'lastName'  => 'CustomerSurname',
                'email'     => 'CustomerEmail',
            ]
        ];
    }

    public static function onPaymentComplete(FoxcommerceOrder $order)
    {
        $passwordInfo = unserialize($order->CustomerTemporaryPassword);
        if(!$passwordInfo) return;

        $currentAlgorithm = Config::inst()->get('Security', 'password_encryption_algorithm');
        Config::inst()->update('Security', 'password_encryption_algorithm', 'none');

        $member                     = new Member();
        $member->FirstName          = $order->CustomerFirstName;
        $member->Surname            = $order->CustomerSurname;
        $member->Email              = $order->CustomerEmail;
        $member->Password           = $passwordInfo['password'];
        $member->PasswordEncryption = 'none';
        $member->write();

        //We need to do these as two steps to prevent the passwords being rehashed
        $member->PasswordEncryption = $passwordInfo['algorithm'];
        $member->Salt               = $passwordInfo['salt'];
        $member->write();

        Config::inst()->update('Security', 'password_encryption_algorithm', $currentAlgorithm);

        $order->MemberID = $member->ID;
        $order->write();
    }
}

class CheckoutStepCustomerDetailsValidator extends RequiredFields
{

    public function php($data)
    {
        if (!Member::currentUser() && isset($data['CustomerPassword'])) {
            $email = $data['CustomerEmail'];
            if (Member::get()->filter('Email', $email)->exists()) {
                /** @var FoxcommerceCheckoutPage $checkoutPage */
                $checkoutPage = FoxcommerceCheckoutPage::get_one('FoxcommerceCheckoutPage');
                $loginLink    = $checkoutPage->Link('account');
                $security     = new Security();
                $resetLink    = $security->Link('lostpassword');

                $message = [
                    'It looks like you already have an account.',
                    'Please <a href="' . $loginLink . '">login</a>, if you have forgotten your password you may <a href="' . $resetLink . '">reset it here</a>.<br />',
                    'You may continue as a guest using this email address if you do not choose to save your details for later use.'
                ];
                $error   = new HTMLText();
                $error->setValue(implode(' ', $message));
                $this->validationError('CustomerEmail', $error, 'validation');
            }
        }

        return parent::php($data);
    }

}