<?php

abstract class CheckoutStep
{

    protected $currentOrder;

    public function __construct(FoxcommerceOrder $order)
    {
        $this->currentOrder = $order;
    }

    /**
     * Push any new fields to the checkout process
     *
     * @param Controller $controller
     * @return FieldList
     */
    abstract public function FormFields(Controller $controller = null);

    /**
     * A pre-loader for the steps
     * If this function returns an instance of SS_HTTPResponse then we will return that response instead
     * of processing the step
     *
     * This is useful for e.g.
     *  - an error message if something isn't setup right
     *  - a redirect to an offsite payment gateway
     *
     * @return null|SS_HTTPResponse
     */
    public function init()
    {
    }

    /**
     * Add any new fields to the validator
     *
     * @return Validator
     */
    public function Validator()
    {
        return new RequiredFields([]);
    }

    /**
     * Has the step been passed?
     * Basically validate that we have all the data we need for this step
     * to be counted as completed
     *
     * @return boolean
     */
    abstract public function hasBeenPassed();

    /**
     * Process the form submission for this step
     *
     * @param array               $data
     * @param CheckoutForm        $form
     * @param SS_HTTPRequest|null $request
     * @return boolean|SS_HTTPResponse
     */
    abstract public function handleSubmission(array $data, CheckoutForm $form, SS_HTTPRequest $request = null);

    /**
     * An array of extra static properties to add to the @see FoxcommerceOrder.
     * All statics are available e.g.
     *  [
     *    'db'      => ['PaymentMethod' => 'Varchar(255)'],
     *    'has_one' => ['OrderImage'    => 'Image']
     *  ]
     *
     * Additionally a 'payment_fields' key may be defined which is an array that maps
     * the silverstripe/omnipay Payment fields to the keys on the Order e.g.
     * ['payment_fields' => [
     *     'firstName'   => 'CustomerFirstName',
     *     'lastName'    => 'CustomerLastName'
     * ]]
     *
     * @return array
     */
    public static function extraStatics()
    {
        return [];
    }

    /**
     * Whether or not this step should be a visible step in the list
     *
     * @return bool
     */
    public function visibleInStepList()
    {
        return true;
    }

    /**
     * Allows us to short circuit the current step, as an example you would
     * short circuit the payment method selection step if there was only on available.
     *
     * @param CheckoutForm $form We pass through the form so you may set a session message if desired
     * @return bool
     */
    public function onAdvance(CheckoutForm $form)
    {
        return false;
    }

    /**
     * Allows us to short circuit the current step, as an example you would
     * short circuit the payment method selection step if there was only on available.
     *
     * @param CheckoutForm $form We pass through the form so you may set a session message if desired
     * @return bool
     */
    public function onRewind(CheckoutForm $form)
    {
        return false;
    }

    public static function onPaymentComplete(FoxcommerceOrder $order){

    }

    public static function config(){
        $config = Config::inst()->get('CheckoutProcess', 'steps')[get_called_class()];
        return is_array($config) ? $config : ['title' => $config];
    }

    public function accessibleByGuest(){
        return false;
    }

    public function updateForm(CheckoutForm $form){

    }

}