<?php

class CheckoutProcess
{

    /**
     * @var SS_HTTPRequest
     */
    protected $currentRequest;

    protected $currentOrder;

    protected static $stepIterator;

    protected static $previousStepIndex;
    protected static $currentStepIndex;
    protected static $nextStepIndex;

    public function __construct(SS_HTTPRequest $currentRequest, FoxcommerceOrder $currentOrder, $forceStepClass = null)
    {
        $this->currentRequest = $currentRequest;
        $this->currentOrder   = $currentOrder;

        $arrayObject        = new ArrayObject(static::stepList());
        self::$stepIterator = $arrayObject->getIterator();
        $action             = $forceStepClass
            ? static::stepList()[$forceStepClass]['url_segment']
            : $this->currentRequest->param('Action');
        $this->setStepIndexesFromAction($action);
    }

    public function setStepIndexesFromIndex($index)
    {
        static::$previousStepIndex = ($index - 1) >= 0 ? ($index - 1) : false;
        static::$currentStepIndex  = $index;
        static::$nextStepIndex     = ($index + 1) <= (self::$stepIterator->count() - 1) ? ($index + 1) : false;
        return true;
    }

    public function setStepIndexesFromAction($action)
    {
        if (!$action) return $this->setStepIndexesFromIndex(0);
        $index = 0;
        foreach (static::stepList() as $step => $data) {
            if ($data['url_segment'] === $action) {
                return $this->setStepIndexesFromIndex($index);
            }
            $index++;
        }

        throw new SS_HTTPResponse_Exception("Checkout action [{$action}] not found", 404);
    }

    /**
     * The payment step is a cutoff point
     * After this step no further steps should edit the order so it's useful for things like
     * A review step, which displays a thanks message and clears the order from the users session
     * but we don't allow the user to go back beyond this step once the order is paid
     *
     * @return string
     */
    public function paymentStep()
    {
        return Config::inst()->get('CheckoutProcess', 'payment_step');
    }

    public function currentOrder()
    {
        return $this->currentOrder;
    }

    public function currentStepIsAfterPaymentStep()
    {
        $paymentStepIndex = array_search($this->paymentStep(), array_keys(self::stepList()));
        if ($paymentStepIndex === false) {
            throw new Exception('CheckoutProcess:payment_step is not a valid step classname');
        }
        if (self::$currentStepIndex > $paymentStepIndex) {
            return true;
        } else {
            return false;
        }
    }

    public function currentStepIsValid()
    {
        $stepClasses = array_keys(self::stepList());
        $iterator    = clone self::$stepIterator;
        $iterator->rewind();

        //First we handle the case where the order is paid so we want to make sure we're
        //Only accessing steps after payment since they shouldn't allow editing (e.g. a summary step)
        $paymentStepIndex = array_search($this->paymentStep(), $stepClasses);
        if ($this->currentOrder->getIsPaid()) {
            if ($this->currentStepIsAfterPaymentStep() || self::$currentStepIndex >= $paymentStepIndex) {
                return true;
            } else {
                self::setStepIndexesFromIndex($paymentStepIndex);
                return false;
            }
        }

        //If the order isn't paid we need to ask each step if it's been passed
        //If we reach our step before one says it failed then we know we're valid
        //If a step says it hasn't been passed then we know that's where the user should be
        foreach ($iterator as $stepData) {

            if ($stepData['classname'] == $stepClasses[self::$currentStepIndex]) {
                return true;
            } else {
                $stepClass = $stepData['classname'];

                /** @var CheckoutStep $step */
                $step = new $stepClass($this->currentOrder);

                if (!$step->hasBeenPassed()) {
                    self::setStepIndexesFromAction($stepData['url_segment']);
                    return false;
                }
            }
        }
    }

    public static function stepList()
    {
        $steps   = Config::inst()->get('CheckoutProcess', 'steps');
        $segment = new URLSegmentFilter();
        array_walk($steps, function (&$value, $key) use ($segment) {
            if (!is_array($value)) $value = ['title' => $value];
            $value['classname']   = $key;
            $value['url_segment'] = $segment->filter(
                $value[isset($value['url_segment']) ? 'url_segment' : 'title']
            );
        });

        return $steps;
    }

    public static function stepByIndex($index)
    {
        $step = static::$stepIterator->offsetGet(array_keys(static::stepList())[$index]);
        return $step;
    }

    public function previousStep()
    {
        if (self::$previousStepIndex === false) return false;
        return self::stepByIndex(self::$previousStepIndex);
    }

    public function currentStep()
    {
        if (self::$currentStepIndex === false) return false;
        return self::stepByIndex(self::$currentStepIndex);
    }

    public function nextStep()
    {
        if (self::$nextStepIndex === false) return false;
        return self::stepByIndex(self::$nextStepIndex);
    }

    public function currentStepIndex()
    {
        return self::$currentStepIndex;
    }

    public function advanceStep()
    {
        if (self::$nextStepIndex === false) return false;
        $this->setStepIndexesFromIndex(self::$nextStepIndex);
        return true;
    }

    public function rewindStep()
    {
        if (self::$previousStepIndex === false) return false;
        $this->setStepIndexesFromIndex(self::$previousStepIndex);
        return true;
    }

    protected function currentStepInstance()
    {
        $stepClass = $this->currentStep()['classname'];
        /** @var CheckoutStep $step */
        return new $stepClass($this->currentOrder);
    }

    public function moveStep($direction, CheckoutForm $form)
    {

        $currentStepIndex = $this->currentStepIndex();
        $direction === 'advance' ? $this->advanceStep() : $this->rewindStep();
        $wantedStepIndex = $this->currentStepIndex();

        //if we haven't moved then we're probably at the end of the train
        if ($currentStepIndex === $wantedStepIndex) return false;

        $step = $this->currentStepInstance();

        if (!$step->visibleInStepList()) {
            switch ($direction) {
                case 'advance':
                    $step->onAdvance($form);
                    $this->moveStep($direction, $form);
                    break;
                case 'rewind':
                    $step->onRewind($form);
                    $this->moveStep($direction, $form);
                    break;
            }
        }
    }

}