<?php

/**
 * Class FoxcommerceProductVariant
 *
 * @property Int $ProductID
 *
 * @method FoxcommerceProduct Product()
 * @method FoxcommerceProductVariantAttributeValueLink[]|HasManyList AttributeValues()
 */
class FoxcommerceProductVariant extends DataObject
{


    protected static $singular_name = 'Product Variant';
    protected static $plural_name   = 'Product Variant';

    protected $is_duplicating          = false;
    protected $is_duplicating_disabled = false;

    protected static $db = [
        'Title'          => 'Varchar(255)',
        'InternalItemID' => 'Varchar',
        'IsEnabled'      => 'Boolean'
    ];

    protected static $extensions = array(
        "Versioned('Live')"
    );

    protected static $has_one = [
        'Product' => 'FoxcommerceProduct'
    ];

    private static $has_many = [
        'AttributeValues' => 'FoxcommerceProductVariantAttributeValueLink'
    ];

    private static $defaults = [
        'IsEnabled' => true
    ];

    private static $summary_fields = [
        'CMSCode'        => 'Code',
        'Title'          => 'Title',
        'AttributesList' => 'Attributes',
        'IsEnabled.Nice' => 'Usable'
    ];

    protected static $cart_summary = [
        'ID',
        'Title',
        'InternalItemID',
        'LastEdited'           => 'RevisionDate',
        'Title',
        'getAttributesSummary' => 'Attributes'
    ];

    public static function enabled()
    {
        return self::get()->filter(['IsEnabled' => true]);
    }

    public function getCMSCode()
    {
        return $this->IsEnabled ? $this->InternalItemID : "{$this->InternalItemID} - COPY";
    }

    public function getCMSFields()
    {

        /** @var FieldList $fields */
        $fields = FieldList::create(TabSet::create('Root'));
        $fields->findOrMakeTab('Root.Main');

        $fields->addFieldsToTab('Root.Main', [
            TextField::create('Title', 'Title'),
            TextField::create('InternalItemID', 'Variation code'),
            HeaderField::create('VariationAttributesHeader', 'Variation Attributes')
        ]);


        if ($this->ProductID) {
            $product = $this->Product();
        } elseif (Session::get('CMSMain.currentPage')) {
            $product = FoxcommerceProduct::get_by_id('FoxcommerceProduct', Session::get('CMSMain.currentPage'));
        } else {
            $product = FoxcommerceProduct::create();
        }


        if ($product->exists()) {
            $attributes = $product->Attributes();
            if ($this->exists()) {
                $attributeValueMap = $this->AttributeValues()->map('AttributeID', 'AttributeValueID')->toArray();
            } else {
                $attributeValueMap = [];
            }

            /** @var FoxcommerceProductAttribute $attribute */
            foreach ($attributes as $attribute) {
                $fieldName   = "VariantAttribute[{$attribute->ID}]";
                $fieldTitle  = $attribute->Title;
                $fieldSource = $attribute->Values()->map('ID', 'MenuTitle')->toArray();
                $fieldValue  = isset($attributeValueMap[$attribute->ID]) ? $attributeValueMap[$attribute->ID] : null;
                $fields->addFieldToTab('Root.Main',
                    DropdownField::create($fieldName, $fieldTitle, $fieldSource, $fieldValue));
            }
        } elseif (!$product->exists()) {
            $productNotFoundMessage      = _t(
                'FoxcommerceProductVariant.ProductNotFoundMessage',
                "We're struggling to determine the product associated with this variant. Please save and we'll see what we can do!"
            );
            $productNotFoundMessageField = LiteralField::create(
                "ProductNotFoundMessageHeader",
                "<p class=\"message warning\">{$productNotFoundMessage}</p>"
            );

            $fields->addFieldToTab('Root.Main', $productNotFoundMessageField);
        }

        $this->extend('updateCMSFields', $fields);

        return $fields;
    }

    public function getSummaryTitle()
    {
        if ($this->InternalItemID) {
            return "{$this->Title} ({$this->InternalItemID})";
        } else {
            return $this->Title;
        }
    }


    protected function onBeforeDuplicate($original)
    {
        $this->IsEnabled      = false;
        $this->is_duplicating = true;
    }

    /**
     * @param self $original
     * @param $doWrite
     * @return void
     */
    protected function onAfterDuplicate($original)
    {
        /** @var FoxcommerceProductVariantAttributeValueLink $attributeValue */
        foreach ($original->AttributeValues() as $attributeValue) {
            /** @var FoxcommerceProductVariantAttributeValueLink $clone */
            $clone            = $attributeValue->duplicate(false);
            $clone->VariantID = $this->ID;
            $clone->write();
        }
    }

    protected function onBeforeWrite()
    {
        parent::onBeforeWrite();
        if (!$this->is_duplicating) {
            $this->IsEnabled = true;
        }
    }

    protected function onAfterWrite()
    {
        parent::onAfterWrite();
        if (!$this->is_duplicating) {
            $this->IsEnabled = true;
        }

        foreach ($this->record as $key => $value) {
            if (preg_match('/VariantAttribute\[(\d+)\]/', $key, $matches)) {
                /** @var FoxcommerceProductVariantAttributeValueLink $attributeLink */
                $attributeLink = $this->AttributeValues()->filter(['AttributeID' => $matches[1]])->first();
                if ($attributeLink && ($attributeLink->AttributeValueID != $value)) {
                    $attributeLink->AttributeValueID = $value;
                    $attributeLink->write();
                } elseif (!$attributeLink) {
                    $attributeLink                   = FoxcommerceProductVariantAttributeValueLink::create();
                    $attributeLink->AttributeID      = $matches[1];
                    $attributeLink->AttributeValueID = $value;
                    $attributeLink->VariantID        = $this->ID;
                    $attributeLink->write();
                }
            }
        }
    }

    public function getAttributesSummary()
    {
        $attributes = [];

        foreach ($this->AttributeValues() as $attr) {
            $attrID  = $attr->AttributeID;
            $valueID = $attr->AttributeValueID;

            if (!isset($attributes[$attrID])) {
                $attributes[$attrID] = [
                    'Attribute' => FoxcommerceHelper::build_cart_summary($attr->Attribute()),
                    'Values'    => []
                ];
            }

            $attributes[$attrID]['Values'][$valueID] = FoxcommerceHelper::build_cart_summary($attr->AttributeValue());
        }

        return $attributes;

    }

    public function getAttributesList()
    {
        $summary = [];

        foreach ($this->AttributeValues() as $attr) {
            $summary[$attr->Attribute()->Title] = $attr->AttributeValue()->Title;
        }

        $comment = [];
        foreach ($summary as $title => $value) {
            $comment[] = "{$title}: {$value}";
        }

        return implode(', ', $comment);
    }

    protected function validate()
    {
        $result = parent::validate();

        if ($this->is_duplicating) {
            return $result;
        }

        $attributeValueMap = [];

        foreach ($this->record as $key => $value) {
            if (preg_match('/VariantAttribute\[(\d+)\]/', $key, $matches)) {
                $attributeValueMap[$matches[1]] = $value;
            }
        }

        //If we're editing outside of the normal interface this is going to be empty
        //at that point it's up to that dev to make sure they're not doing anything bad...
        if(empty($attributeValueMap)){
            return $result;
        }

        /** @var HasManyList $variants */
        $variants = $this->Product()->Variants();

        $variantIDs = $variants->filter(['IsEnabled' => true])->exclude(['ID' => $this->ID])
            ->map('ID', 'Title')->toArray();

        $map = FoxcommerceProductVariantAttributeValueLink::get()
            ->filter(['VariantID' => array_keys($variantIDs)]);

        $variantAttributeMap = [];

        /** @var FoxcommerceProductVariantAttributeValueLink $link */
        foreach ($map as $link) {
            if (!isset($variantAttributeMap[$link->VariantID])) {
                $variantAttributeMap[$link->VariantID] = [];
            }

            $variantAttributeMap[$link->VariantID][$link->AttributeID] = $link->AttributeValueID;
        }

        $identicalVariant = array_filter($variantAttributeMap, function ($value) use ($attributeValueMap) {
            return empty(array_diff_assoc($attributeValueMap, $value));
        });

        if (!empty($identicalVariant)) {
            $duplicateVariationTitle = $variantIDs[key($identicalVariant)];
            $result->error("This variants attributes duplicate those of variant \"{$duplicateVariationTitle}\" and cannot be saved!");
        }

        return $result;
    }

    public function canCreate($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canView($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canEdit($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }

    public function canDelete($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }
}
