<?php

/**
 * Class FoxcommerceTaxRateOrderModifier
 *
 * The calculations for when products are marked as GST inclusive
 * - Tax Amount:          InclusivePrice * TaxRate
 * - Price Without Tax:   InclusivePrice * (1 - TaxRate)
 * - Price Including Tax: InclusivePrice
 *
 * Given a TaxRate of 0.25 (25%) and an input of $125 we would expect
 * - Tax Amount:          25
 * - Price Without Tax:   100
 * - Price Including Tax: 125
 *
 * The calculations for when products are marked as GST exclusive
 * - Tax Amount:          ExclusivePrice / (1 - TaxRate)
 * - Price Without Tax:   ExclusivePrice
 * - Price Including Tax: ExclusivePrice + TaxAmount
 *
 * Given a TaxRate of 0.25 (25%) and an input of $80 we would expect
 * - Tax Amount:          20
 * - Price Without Tax:   80
 * - Price Including Tax: 100
 *
 */
class FoxcommerceTaxRateOrderModifier extends FoxcommerceOrderModifier
{

    protected static $defaults = [
        'ApplicationTrigger'   => 'Automatic',
        'RecalculationTrigger' => 'Automatic',
        'ApplicationStage'     => 'Total',
        'ModifierType'         => 'Addition',
        'Exclusive'            => true,
        'IsApplied'            => true,
        'VisibleIfEmpty'       => false
    ];

    public function getTitle()
    {
        if (!$this->order || !$this->order->Country()->TaxRateID) {
            $title = parent::getTitle();
            $this->extend('updateTitle', $title);
            return $title;
        }
        if ($this->is('Inclusive')) {
            $title = 'Including ' . $this->order->Country()->TaxRate()->Code . ' of';
            $this->extend('updateTitle', $title);
            return $title;
        }

        $title = $this->order->Country()->TaxRate()->Code;
        $this->extend('updateTitle', $title);
        return $title;
    }

    public function calculateForDisplay(FoxcommerceOrder $order, $numberOnly = false)
    {
        $hasTax = $order->Country()->TaxRateID;

        if ($hasTax) {
            $rate = $order->Country()->TaxRate();
            if ($this->is('Inclusive')) {
                $amountWithoutTax = $rate->subtractFrom($order->Subtotal);
                $amount           = BCMath::subtract($order->Subtotal, $amountWithoutTax);
                $this->extend('updateForDisplay', $amount);
                return $amount;
            } elseif ($this->is('Exclusive')) {
                $amount = BCMath::multiply($order->Subtotal, $rate->Rate);
                $this->extend('updateForDisplay', $amount);
                return $amount;
            }
        }

        $amount = '';
        $this->extend('updateForDisplay', $amount);

        return ($numberOnly ? (is_numeric($amount) ? $amount : "0") : $amount);
    }

    public function calculateForOrder(FoxcommerceOrder $order)
    {
        if ($this->is('Exclusive')) {
            return $this->calculateForDisplay($order, true);
        }
    }

    public function is($type)
    {
        return strtolower(FoxcommerceConfig::current()->DisplayTaxStyle) === strtolower($type);
    }
}