<?php

/**
 * Class FoxcommerceCountry
 *
 * @property string  $Title
 * @property string  $Code
 * @property float   $Rate
 * @property boolean $Enabled
 * @method DataList|FoxcommerceCountry[] Countries()
 */
class FoxcommerceTaxRate extends DataObject
{

    protected static $singular_name = 'Tax Rate';
    protected static $plural_name   = 'Tax Rates';

    protected static $db = [
        'Title'   => 'Varchar(255)',
        'Code'    => 'Varchar',
        'Rate'    => 'Percentage',
        'Enabled' => 'Boolean'
    ];

    private static $has_many = [
        'Countries' => 'FoxcommerceCountry'
    ];

    private static $defaults = [
        'Enabled' => true
    ];

    private static $summary_fields = [
        'Title'         => 'Title',
        'Description'   => 'Description',
        'CountriesList' => 'Applicable Countries',
        'Enabled.Nice'  => 'Enabled'
    ];

    private static $default_sort = 'Code ASC';

    public function subtractFrom($amount)
    {
        return BCMath::divide($amount, BCMath::add("1", $this->Rate));
    }

    public function addTo($amount)
    {
        return BCMath::add($amount, BCMath::multiply($amount, $this->Rate));
    }

    public function getCMSFields()
    {

        $countries = FoxcommerceCountry::get('FoxcommerceCountry')->map()->toArray();

        $fields = new FieldList([
            TextField::create('Title')->setPlaceholder('Goods and Services Tax'),
            TextField::create('Code')->setPlaceholder('GST'),
            TextField::create('Rate')->setPlaceholder('0.15')
                ->setDescription(_t(
                    'FoxcommerceTaxRate.TaxRateHelpMessage',
                    'Enter the rate as a decimal e.g. 15% should be entered as 0.15. Up to four decimal places.'
                )),
            ListboxField::create('Countries', 'Countries', $countries)->setMultiple(true),
            DropdownField::create('Enabled', 'Enabled', [1 => 'Enabled', 0 => 'Disabled']),
        ]);

        $this->extend('updateCMSFields', $fields);

        return $fields;
    }

    public function getDescription()
    {
        if ($this->exists()) {
            return "{$this->Code} at {$this->dbObject('Rate')->Nice()}";
        }
        return '(none)';
    }

    public function getCountriesList()
    {
        return implode(', ', $this->Countries()->column('Title'));
    }
    
    public function canCreate($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }
 
    public function canView($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }
 
    public function canEdit($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }
 
    public function canDelete($member = null)
    {
        return Permission::check('CMS_ACCESS_CMSMain', 'any', $member);
    }


}